<?php

namespace Drupal\resource_conflict\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Database\Connection;
use Drupal\Core\Datetime\DrupalDateTime;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\datetime\Plugin\Field\FieldType\DateTimeItemInterface;
use Drupal\field\FieldConfigInterface;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\FieldStorageDefinitionInterface;
use Drupal\node\NodeInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\resource_conflict\Event\ConflictDetectedEvent;
use Drupal\resource_conflict\Event\ResourceConflictEvents;
use Symfony\Contracts\EventDispatcher\EventDispatcherInterface;
use Drupal\date_recur\DateRecurOccurrences;
use Drupal\date_recur\Plugin\Field\FieldType\DateRecurItem;
use Drupal\smart_date\Plugin\Field\FieldType\SmartDateItem;

/**
 * Provides utilities for the Resource Conflict module.
 */
class ResourceConflictManager {

  use StringTranslationTrait;

  /**
   * The name of the managed configuration object.
   */
  private const string SETTINGS_NAME = 'resource_conflict.settings';

  /**
   * Maximum occurrences generated for infinite Recurring Dates values.
   */
  private const int DATE_RECUR_MAX_OCCURRENCES = 500;

  /**
   * Constructs the manager.
   */
  public function __construct(
    private readonly ConfigFactoryInterface $configFactory,
    private readonly EntityTypeManagerInterface $entityTypeManager,
    private readonly EntityFieldManagerInterface $entityFieldManager,
    private readonly ModuleHandlerInterface $moduleHandler,
    private readonly MessengerInterface $messenger,
    private readonly LoggerChannelFactoryInterface $loggerFactory,
    private readonly EventDispatcherInterface $eventDispatcher,
    private readonly Connection $database,
  ) {}

  /**
   * Returns saved settings for all bundles.
   */
  public function getAllSettings(): array {
    $raw = $this->configFactory->get(self::SETTINGS_NAME)->get('bundles') ?? [];
    return $this->normalizeSettings(is_array($raw) ? $raw : []);
  }

  /**
   * Returns settings for a specific bundle.
   */
  public function getBundleSettings(string $bundle): array {
    $all = $this->getAllSettings();
    if (!isset($all[$bundle])) {
      return [];
    }
    return $all[$bundle] + [
      'default_form_error' => TRUE,
    ];
  }

  /**
   * Determines whether the provided bundle is configured.
   */
  public function isBundleEnabled(string $bundle): bool {
    $settings = $this->getBundleSettings($bundle);
    return !empty($settings['enabled']) && !empty($settings['field_name']);
  }

  /**
   * Saves settings for a bundle.
   */
  public function saveBundleSettings(string $bundle, array $values): void {
    $all = $this->getAllSettings();

    if (empty($values['enabled']) || empty($values['field_name'])) {
      unset($all[$bundle]);
      $this->persistSettings($all);
      return;
    }

    $all[$bundle] = [
      'enabled' => TRUE,
      'field_name' => $values['field_name'],
      'restrict_to_bundle' => !empty($values['restrict_to_bundle']),
      'start_buffer' => trim($values['start_buffer'] ?? ''),
      'end_buffer' => trim($values['end_buffer'] ?? ''),
      'default_form_error' => !empty($values['default_form_error']),
    ];

    $this->persistSettings($all);
  }

  /**
   * Removes any settings for the provided bundle.
   */
  public function deleteBundleSettings(string $bundle): void {
    $all = $this->getAllSettings();
    if (isset($all[$bundle])) {
      unset($all[$bundle]);
      $this->persistSettings($all);
    }
  }

  /**
   * Builds options for compatible fields on the bundle.
   */
  public function getEligibleFieldOptions(string $bundle): array {
    $definitions = $this->entityFieldManager->getFieldDefinitions('node', $bundle);
    $options = [];
    foreach ($definitions as $definition) {
      if ($definition->getFieldStorageDefinition()->isBaseField()) {
        continue;
      }
      if ($this->fieldIsCompatible($definition->getFieldStorageDefinition())) {
        $options[$definition->getName()] = $definition->getLabel();
      }
    }

    natcasesort($options);
    return $options;
  }

  /**
   * Determines if the provided field configuration is compatible.
   */
  public function fieldIsCompatible(FieldStorageDefinitionInterface $definition): bool {
    return match ($definition->getType()) {
      'daterange' => TRUE,
      'date_recur' => $this->moduleHandler->moduleExists('date_recur'),
      'smartdate' => $this->moduleHandler->moduleExists('smart_date'),
      default => FALSE,
    };
  }

  /**
   * Handles a field deletion event.
   */
  public function handleFieldConfigDeletion(FieldConfigInterface $field): void {
    $bundle = $field->getTargetBundle();
    $settings = $this->getBundleSettings($bundle);
    if (!empty($settings) && $settings['field_name'] === $field->getName()) {
      $this->deleteBundleSettings($bundle);
      $this->messenger->addWarning($this->t('Resource conflict detection was disabled for %type because the configured field was removed.', ['%type' => $bundle]));
    }
  }

  /**
   * Handles a field update.
   */
  public function handleFieldConfigUpdate(FieldConfigInterface $field): void {
    $bundle = $field->getTargetBundle();
    $settings = $this->getBundleSettings($bundle);
    if (empty($settings) || $settings['field_name'] !== $field->getName()) {
      return;
    }

    if (!$this->fieldIsCompatible($field->getFieldStorageDefinition())) {
      $this->deleteBundleSettings($bundle);
      $this->messenger->addWarning($this->t('Resource conflict detection was disabled for %type because the configured field is no longer a compatible date range field.', ['%type' => $bundle]));
    }
  }

  /**
   * Loads conflicts for the provided node.
   *
   * @return \Drupal\node\NodeInterface[]
   *   The conflicting nodes keyed by nid.
   */
  public function getConflicts(NodeInterface $node): array {
    $nids = $this->getConflictingNodeIds($node);
    if (empty($nids)) {
      return [];
    }

    /** @var \Drupal\node\NodeStorageInterface $storage */
    $storage = $this->entityTypeManager->getStorage('node');
    $conflicts = $storage->loadMultiple($nids);
    foreach ($conflicts as $conflict) {
      $this->eventDispatcher->dispatch(new ConflictDetectedEvent($node, $conflict), ResourceConflictEvents::CONFLICT_DETECTED);
    }
    $this->moduleHandler->invokeAll('resource_conflict_conflicts_detected', [$node, $conflicts]);

    return $conflicts;
  }

  /**
   * Gathers time spans from the configured date range field.
   */
  protected function getTimeSpans(NodeInterface $node): array {
    $settings = $this->getBundleSettings($node->bundle());
    $field_name = $settings['field_name'] ?? NULL;
    if (!$field_name || !$node->hasField($field_name)) {
      return [];
    }

    $items = $node->get($field_name);
    if ($items->isEmpty()) {
      return [];
    }

    $definition = $items->getFieldDefinition();
    return match ($definition->getType()) {
      'date_recur' => $this->moduleHandler->moduleExists('date_recur') ? $this->getDateRecurTimeSpans($items, $settings) : [],
      'smartdate' => $this->moduleHandler->moduleExists('smart_date') ? $this->getSmartDateTimeSpans($items, $settings) : [],
      default => $this->getDateRangeTimeSpans($items, $settings),
    };
  }

  /**
   * Builds time spans from a standard Date range field.
   */
  protected function getDateRangeTimeSpans(FieldItemListInterface $items, array $settings): array {
    $time_spans = [];

    foreach ($items as $item) {
      /** @var \Drupal\Core\Field\FieldItemInterface $item */
      if ($item->isEmpty()) {
        continue;
      }
      $start_source = $item->get('start_date')->getValue();
      $end_source = $item->get('end_date')->getValue();
      if (!$start_source instanceof DrupalDateTime || !$end_source instanceof DrupalDateTime) {
        continue;
      }

      $span = $this->buildTimeSpan($start_source, $end_source, $settings);
      $time_spans[] = $span;
    }

    return $time_spans;
  }

  /**
   * Builds time spans from a Smart Date (`smart_date`) value.
   */
  protected function getSmartDateTimeSpans(FieldItemListInterface $items, array $settings): array {
    $time_spans = [];

    foreach ($items as $item) {
      if (!$item instanceof SmartDateItem) {
        continue;
      }
      $start_source = $item->get('start_time')->getValue();
      $end_source = $item->get('end_time')->getValue();

      if (!$start_source instanceof DrupalDateTime || !$end_source instanceof DrupalDateTime) {
        continue;
      }

      $time_spans[] = $this->buildSmartDateSpan($start_source, $end_source, $settings);
    }

    return $time_spans;
  }

  /**
   * Builds time spans from a Recurring Dates Field (`date_recur`) value.
   */
  protected function getDateRecurTimeSpans(FieldItemListInterface $items, array $settings): array {
    $time_spans = [];
    if (!class_exists(DateRecurItem::class)) {
      return $time_spans;
    }

    foreach ($items as $item) {
      if (!$item instanceof DateRecurItem) {
        continue;
      }

      try {
        $helper = $item->getHelper();
      }
      catch (\Throwable) {
        continue;
      }

      $limit = $helper->isInfinite() ? self::DATE_RECUR_MAX_OCCURRENCES : NULL;

      try {
        $occurrences = $helper->getOccurrences(NULL, NULL, $limit);
      }
      catch (\Throwable) {
        continue;
      }

      foreach ($occurrences as $occurrence) {
        $start = DrupalDateTime::createFromDateTime($occurrence->getStart());
        $end = DrupalDateTime::createFromDateTime($occurrence->getEnd());
        $time_spans[] = $this->buildTimeSpan($start, $end, $settings);
      }
    }

    return $time_spans;
  }

  /**
   * Applies configured buffers and formats a time span for storage.
   */
  protected function buildTimeSpan(DrupalDateTime $start, DrupalDateTime $end, array $settings): array {
    $start_clone = clone $start;
    $end_clone = clone $end;

    if (!empty($settings['start_buffer'])) {
      $start_clone = $this->applyBuffer($start_clone, $settings['start_buffer']);
    }

    if (!empty($settings['end_buffer'])) {
      $end_clone = $this->applyBuffer($end_clone, $settings['end_buffer']);
    }

    return [
      'start' => $start_clone->format(DateTimeItemInterface::DATETIME_STORAGE_FORMAT),
      'end' => $end_clone->format(DateTimeItemInterface::DATETIME_STORAGE_FORMAT),
    ];
  }

  /**
   * Formats a Smart Date span using timestamps for storage queries.
   */
  protected function buildSmartDateSpan(DrupalDateTime $start, DrupalDateTime $end, array $settings): array {
    $start_clone = clone $start;
    $end_clone = clone $end;

    if (!empty($settings['start_buffer'])) {
      $start_clone = $this->applyBuffer($start_clone, $settings['start_buffer']);
    }

    if (!empty($settings['end_buffer'])) {
      $end_clone = $this->applyBuffer($end_clone, $settings['end_buffer']);
    }

    return [
      'start' => (int) $start_clone->getTimestamp(),
      'end' => (int) $end_clone->getTimestamp(),
    ];
  }

  /**
   * Applies a buffer string to the provided date.
   */
  protected function applyBuffer(DrupalDateTime $date, string $modifier): DrupalDateTime {
    $clone = clone $date;
    try {
      $clone->modify($modifier);
      return $clone;
    }
    catch (\Exception $e) {
      $this->loggerFactory->get('resource_conflict')->error('Failed to apply buffer modifier "@modifier" to date "@date": @message', [
        '@modifier' => $modifier,
        '@date' => $date->format(DateTimeItemInterface::DATETIME_STORAGE_FORMAT),
        '@message' => $e->getMessage(),
      ]);
      return $date;
    }
  }

  /**
   * Finds node IDs that overlap with the provided node.
   *
   * @return int[]
   *   A list of node IDs.
   */
  protected function getConflictingNodeIds(NodeInterface $node): array {
    if (!$this->isBundleEnabled($node->bundle())) {
      return [];
    }

    $settings = $this->getBundleSettings($node->bundle());
    $field_name = $settings['field_name'] ?? NULL;
    if (!$field_name || !$node->hasField($field_name)) {
      return [];
    }

    $items = $node->get($field_name);
    if ($items->isEmpty()) {
      return [];
    }

    $time_spans = $this->getTimeSpans($node);
    if (!$time_spans) {
      return [];
    }

    $bundles = $settings['restrict_to_bundle'] ? [$node->bundle()] : $this->getBundlesUsingField($field_name);
    if (empty($bundles)) {
      $bundles = [$node->bundle()];
    }

    $definition = $items->getFieldDefinition();
    if ($definition->getType() === 'date_recur' && $this->moduleHandler->moduleExists('date_recur')) {
      return $this->getDateRecurConflictingNodeIds($node, $items, $time_spans, $bundles);
    }
    if ($definition->getType() === 'smartdate' && $this->moduleHandler->moduleExists('smart_date')) {
      return $this->getSmartDateConflictingNodeIds($node, $field_name, $time_spans, $bundles, $settings);
    }

    return $this->getDateRangeConflictingNodeIds($node, $field_name, $time_spans, $bundles);
  }

  /**
   * Finds conflicting nodes for standard Date range fields.
   */
  protected function getDateRangeConflictingNodeIds(NodeInterface $node, string $field_name, array $time_spans, array $bundles): array {
    /** @var \Drupal\node\NodeStorageInterface $storage */
    $storage = $this->entityTypeManager->getStorage('node');
    $nids = [];

    foreach ($time_spans as $span) {
      $query = $storage->getQuery();
      // Access validation happens higher up in the form/presave hooks.
      // @phpstan-ignore-next-line Drupal's PHPStan rules flag literal FALSE unnecessarily here.
      $query->accessCheck(FALSE);
      $query->condition('type', $bundles, 'IN');
      if ($node->id()) {
        $query->condition('nid', $node->id(), '<>');
      }

      $conditions = $query->orConditionGroup()
        ->condition(
          $query->andConditionGroup()
            ->condition("$field_name.value", $span['start'], '<=')
            ->condition("$field_name.end_value", $span['start'], '>')
        )
        ->condition(
          $query->andConditionGroup()
            ->condition("$field_name.value", $span['end'], '<')
            ->condition("$field_name.end_value", $span['end'], '>=')
        )
        ->condition(
          $query->andConditionGroup()
            ->condition("$field_name.value", $span['start'], '>=')
            ->condition("$field_name.end_value", $span['end'], '<=')
        );

      $query->condition($conditions);

      $result = $query->execute();
      if (!empty($result)) {
        $nids = array_merge($nids, $result);
      }
    }

    $unique = array_unique($nids);
    return array_values(array_map(static fn ($value) => (int) $value, $unique));
  }

  /**
   * Finds conflicting nodes for Smart Date fields using timestamp comparisons.
   */
  protected function getSmartDateConflictingNodeIds(NodeInterface $node, string $field_name, array $time_spans, array $bundles, array $settings): array {
    /** @var \Drupal\node\NodeStorageInterface $storage */
    $storage = $this->entityTypeManager->getStorage('node');
    $nids = [];

    foreach ($time_spans as $span) {
      $query = $storage->getQuery();
      // Access validation happens higher up in the form/presave hooks.
      // @phpstan-ignore-next-line Drupal's PHPStan rules flag literal FALSE unnecessarily here.
      $query->accessCheck(FALSE);
      $query->condition('type', $bundles, 'IN');
      if ($node->id()) {
        $query->condition('nid', $node->id(), '<>');
      }

      $conditions = $query->orConditionGroup()
        ->condition(
          $query->andConditionGroup()
            ->condition("$field_name.value", $span['start'], '<=')
            ->condition("$field_name.end_value", $span['start'], '>')
        )
        ->condition(
          $query->andConditionGroup()
            ->condition("$field_name.value", $span['end'], '<')
            ->condition("$field_name.end_value", $span['end'], '>=')
        )
        ->condition(
          $query->andConditionGroup()
            ->condition("$field_name.value", $span['start'], '>=')
            ->condition("$field_name.end_value", $span['end'], '<=')
        );

      $query->condition($conditions);

      $result = $query->execute();
      if (!empty($result)) {
        $nids = array_merge($nids, $result);
      }
    }

    $unique = array_unique($nids);
    return array_values(array_map('intval', $unique));
  }

  /**
   * Finds conflicting nodes for Recurring Dates Field values via occurrences.
   */
  protected function getDateRecurConflictingNodeIds(NodeInterface $node, FieldItemListInterface $items, array $time_spans, array $bundles): array {
    if (!class_exists(DateRecurOccurrences::class)) {
      return [];
    }

    $definition = $items->getFieldDefinition();
    $storage_definition = $definition->getFieldStorageDefinition();
    $table = DateRecurOccurrences::getOccurrenceCacheStorageTableName($storage_definition);
    if (!$this->database->schema()->tableExists($table)) {
      return [];
    }

    $start_column = $definition->getName() . '_value';
    $end_column = $definition->getName() . '_end_value';
    $nids = [];

    foreach ($time_spans as $span) {
      $query = $this->database->select($table, 'occ');
      $query->addField('occ', 'entity_id');
      $query->join('node_field_data', 'nfd', 'nfd.nid = occ.entity_id');
      $query->condition('nfd.type', $bundles, 'IN');

      if ($node->id()) {
        $query->condition('occ.entity_id', $node->id(), '<>');
      }

      $conditions = $query->orConditionGroup()
        ->condition(
          $query->andConditionGroup()
            ->condition("occ.$start_column", $span['start'], '<=')
            ->condition("occ.$end_column", $span['start'], '>')
        )
        ->condition(
          $query->andConditionGroup()
            ->condition("occ.$start_column", $span['end'], '<')
            ->condition("occ.$end_column", $span['end'], '>=')
        )
        ->condition(
          $query->andConditionGroup()
            ->condition("occ.$start_column", $span['start'], '>=')
            ->condition("occ.$end_column", $span['end'], '<=')
        );

      $query->condition($conditions);

      $result = $query->execute()->fetchCol();
      if (!empty($result)) {
        $nids = array_merge($nids, array_map('intval', $result));
      }
    }

    $unique = array_unique($nids);
    return array_values($unique);
  }

  /**
   * Returns bundles that share a field configuration.
   */
  protected function getBundlesUsingField(string $field_name): array {
    $bundles = [];
    foreach ($this->getAllSettings() as $bundle => $settings) {
      if (!empty($settings['enabled']) && ($settings['field_name'] ?? NULL) === $field_name) {
        $bundles[] = $bundle;
      }
    }
    return $bundles;
  }

  /**
   * Normalizes stored settings to an associative array keyed by bundle.
   */
  protected function normalizeSettings(array $raw): array {
    $normalized = [];
    foreach ($raw as $entry) {
      if (!is_array($entry) || empty($entry['bundle'])) {
        continue;
      }
      $bundle = $entry['bundle'];
      unset($entry['bundle']);
      $normalized[$bundle] = $entry;
    }

    return $normalized;
  }

  /**
   * Persists bundle settings to configuration.
   */
  protected function persistSettings(array $settings): void {
    $list = [];
    foreach ($settings as $bundle => $values) {
      $list[] = ['bundle' => $bundle] + $values;
    }

    $this->configFactory->getEditable(self::SETTINGS_NAME)
      ->set('bundles', $list)
      ->save();
  }

}
