<?php

namespace Drupal\resource_conflict_test\EventSubscriber;

use Drupal\resource_conflict\Event\ConflictDetectedEvent;
use Drupal\resource_conflict\Event\NodeValidationEvent;
use Drupal\resource_conflict\Event\ResourceConflictEvents;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

/**
 * Captures Resource Conflict events for assertions.
 */
class ResourceConflictTestSubscriber implements EventSubscriberInterface {

  /**
   * Recorded node validation labels.
   *
   * @var string[]
   */
  protected static array $validationEvents = [];

  /**
   * Recorded conflict pairs.
   *
   * @var array[]
   */
  protected static array $conflictEvents = [];

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    return [
      ResourceConflictEvents::NODE_VALIDATION => 'onNodeValidation',
      ResourceConflictEvents::CONFLICT_DETECTED => 'onConflictDetected',
    ];
  }

  /**
   * Reacts to node validation events.
   */
  public function onNodeValidation(NodeValidationEvent $event): void {
    self::$validationEvents[] = $event->getNode()->label();
  }

  /**
   * Reacts to conflict detected events.
   */
  public function onConflictDetected(ConflictDetectedEvent $event): void {
    self::$conflictEvents[] = [
      'node' => $event->getNode()->id(),
      'conflict' => $event->getConflictingNode()->id(),
    ];
  }

  /**
   * Resets captured events.
   */
  public static function reset(): void {
    self::$validationEvents = [];
    self::$conflictEvents = [];
  }

  /**
   * Returns recorded validation events.
   */
  public static function getValidationEvents(): array {
    return self::$validationEvents;
  }

  /**
   * Returns recorded conflict events.
   */
  public static function getConflictEvents(): array {
    return self::$conflictEvents;
  }

}
