<?php

declare(strict_types=1);

namespace Drupal\Tests\resource_conflict\Functional;

use Drupal\Core\Field\FieldStorageDefinitionInterface;
use Drupal\field\Entity\FieldConfig;
use Drupal\field\Entity\FieldStorageConfig;
use Drupal\node\Entity\Node;
use Drupal\node\Entity\NodeType;
use Drupal\Tests\BrowserTestBase;

/**
 * Functional regression coverage for resource conflict validation.
 *
 * @group resource_conflict
 */
class ResourceConflictUiTest extends BrowserTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'node',
    'user',
    'field',
    'datetime',
    'datetime_range',
    'resource_conflict',
  ];

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * Name of the test content type.
   */
  protected const BUNDLE = 'rc_page';

  /**
   * Machine name of the configured field.
   */
  protected const FIELD_NAME = 'field_conflict_dates';

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    NodeType::create([
      'type' => static::BUNDLE,
      'name' => 'Resource conflict page',
    ])->save();

    FieldStorageConfig::create([
      'entity_type' => 'node',
      'field_name' => static::FIELD_NAME,
      'type' => 'daterange',
      'cardinality' => FieldStorageDefinitionInterface::CARDINALITY_UNLIMITED,
    ])->save();
    FieldConfig::create([
      'entity_type' => 'node',
      'field_name' => static::FIELD_NAME,
      'bundle' => static::BUNDLE,
      'label' => 'Conflict dates',
      'required' => TRUE,
    ])->save();

    $this->config('system.date')->set('timezone.default', 'UTC')->save();

    $form_display = $this->container->get('entity_display.repository')->getFormDisplay('node', static::BUNDLE, 'default');
    $form_display->setComponent(static::FIELD_NAME, [
      'type' => 'daterange_default',
    ])->save();

    $account = $this->drupalCreateUser([
      "create " . static::BUNDLE . " content",
      "edit any " . static::BUNDLE . " content",
    ]);
    $this->drupalLogin($account);
  }

  /**
   * Ensures existing conflicts can be corrected after enabling the module.
   */
  public function testEditingConflictResolves(): void {
    [$first, $second] = $this->createConflictingPair();
    $this->enableResourceConflict();

    $edit = $this->buildDateRangeInput('2024-08-01T11:00:00', '2024-08-01T12:00:00');
    $this->drupalGet('node/' . $second->id() . '/edit');
    $this->submitForm($edit, 'Save');
    $this->assertSession()->pageTextContains('has been updated');
    $node = Node::load($second->id());
    $this->assertNotNull($node);
    $value = $node->get(static::FIELD_NAME)->first();
    $this->assertNotNull($value);
    /** @var \Drupal\datetime_range\Plugin\Field\FieldType\DateRangeItem $value */
    $this->assertEquals('2024-08-01T11:00:00', $value->getValue()['value']);
  }

  /**
   * Ensures editing a node to create a conflict triggers a form error.
   */
  public function testEditingIntroducesConflictSetsFormError(): void {
    $this->enableResourceConflict();
    $first = $this->createNodeWithDates('Morning', '2024-09-01T09:00:00', '2024-09-01T10:00:00');
    $second = $this->createNodeWithDates('Afternoon', '2024-09-01T11:00:00', '2024-09-01T12:00:00');

    $edit = $this->buildDateRangeInput('2024-09-01T09:30:00', '2024-09-01T10:30:00');
    $this->drupalGet('node/' . $second->id() . '/edit');
    $this->submitForm($edit, 'Save');
    $this->assertSession()->pageTextContains('This entry conflicts with the following content');
    $this->assertSession()->statusMessageContains('This entry conflicts with the following content', 'error');
    $this->assertSession()->linkExists($first->label());

    $second = Node::load($second->id());
    $this->assertNotNull($second);
    $value = $second->get(static::FIELD_NAME)->first();
    $this->assertNotNull($value);
    /** @var \Drupal\datetime_range\Plugin\Field\FieldType\DateRangeItem $value */
    $this->assertEquals('2024-09-01T11:00:00', $value->getValue()['value']);
  }

  /**
   * Creates two nodes whose times overlap.
   */
  protected function createConflictingPair(): array {
    $first = $this->createNodeWithDates('Original', '2024-08-01T09:00:00', '2024-08-01T10:00:00');
    $second = $this->createNodeWithDates('Conflicting', '2024-08-01T09:30:00', '2024-08-01T10:30:00');
    return [$first, $second];
  }

  /**
   * Creates a test node with the configured date range.
   */
  protected function createNodeWithDates(string $title, string $start, string $end): Node {
    $node = Node::create([
      'type' => static::BUNDLE,
      'title' => $title,
      static::FIELD_NAME => [
        'value' => $start,
        'end_value' => $end,
      ],
    ]);
    $node->save();
    return $node;
  }

  /**
   * Builds form values for the date range widget.
   */
  protected function buildDateRangeInput(string $start, string $end): array {
    [$start_date, $start_time] = explode('T', $start);
    [$end_date, $end_time] = explode('T', $end);
    return [
      static::FIELD_NAME . '[0][value][date]' => $start_date,
      static::FIELD_NAME . '[0][value][time]' => $start_time,
      static::FIELD_NAME . '[0][end_value][date]' => $end_date,
      static::FIELD_NAME . '[0][end_value][time]' => $end_time,
    ];
  }

  /**
   * Enables resource conflict checking for the test bundle.
   */
  protected function enableResourceConflict(array $overrides = []): void {
    $settings = [
      'enabled' => TRUE,
      'field_name' => static::FIELD_NAME,
      'restrict_to_bundle' => FALSE,
      'start_buffer' => '',
      'end_buffer' => '',
      'default_form_error' => TRUE,
    ];
    $this->container->get('resource_conflict.manager')->saveBundleSettings(static::BUNDLE, $overrides + $settings);
  }

}
