<?php

declare(strict_types=1);

namespace Drupal\Tests\resource_conflict\Kernel;

use Drupal\Core\Field\FieldStorageDefinitionInterface;
use Drupal\Core\Datetime\DrupalDateTime;
use Drupal\field\Entity\FieldConfig;
use Drupal\field\Entity\FieldStorageConfig;
use Drupal\KernelTests\KernelTestBase;
use Drupal\node\Entity\Node;
use Drupal\smart_date_recur\Entity\SmartDateRule;
use Drupal\node\Entity\NodeType;
use Drupal\resource_conflict\Service\ResourceConflictManager;
use Drupal\user\Entity\User;

/**
 * Verifies Smart Date (smart_date) fields participate in conflict detection.
 *
 * @group resource_conflict
 */
class SmartDateIntegrationTest extends KernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'user',
    'field',
    'text',
    'node',
    'datetime',
    'datetime_range',
    'resource_conflict',
    'smart_date',
    'smart_date_recur',
  ];

  /**
   * The manager under test.
   */
  protected ResourceConflictManager $manager;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installEntitySchema('user');
    $this->installEntitySchema('node');
    $this->installEntitySchema('smart_date_rule');
    $this->installSchema('system', ['sequences']);
    $this->installConfig([
      'node',
      'resource_conflict',
      'smart_date',
      'smart_date_recur',
    ]);

    User::create(['name' => 'admin'])->save();

    NodeType::create([
      'type' => 'booking',
      'name' => 'Booking',
    ])->save();

    FieldStorageConfig::create([
      'entity_type' => 'node',
      'field_name' => 'field_smart_dates',
      'type' => 'smartdate',
      'cardinality' => FieldStorageDefinitionInterface::CARDINALITY_UNLIMITED,
      'settings' => [],
    ])->save();

    FieldConfig::create([
      'entity_type' => 'node',
      'field_name' => 'field_smart_dates',
      'bundle' => 'booking',
      'label' => 'Smart dates',
      'required' => TRUE,
    ])->save();

    $this->manager = $this->container->get('resource_conflict.manager');
    $this->manager->saveBundleSettings('booking', [
      'enabled' => TRUE,
      'field_name' => 'field_smart_dates',
      'restrict_to_bundle' => TRUE,
      'start_buffer' => '',
      'end_buffer' => '',
    ]);
  }

  /**
   * Ensures Smart Date fields are exposed as eligible options.
   */
  public function testSmartDateFieldListedAsEligible(): void {
    $options = $this->manager->getEligibleFieldOptions('booking');
    $this->assertArrayHasKey('field_smart_dates', $options);
  }

  /**
   * Ensures Smart Date (smart_date) values produce overlaps.
   */
  public function testSmartDateConflictsDetected(): void {
    $node_a = $this->createSmartDateNode('Original', '2024-07-01T10:00:00', '2024-07-01T11:00:00');
    $node_a->save();

    $node_b = $this->createSmartDateNode('Overlap', '2024-07-01T10:30:00', '2024-07-01T11:30:00');
    $item = $node_b->get('field_smart_dates')->first();
    $this->assertInstanceOf(DrupalDateTime::class, $item->get('start_time')->getValue());
    $this->assertInstanceOf(DrupalDateTime::class, $item->get('end_time')->getValue());
    $conflicts = $this->manager->getConflicts($node_b);

    $this->assertCount(1, $conflicts);
    $this->assertSame($node_a->id(), reset($conflicts)->id());
  }

  /**
   * Ensures multi-value Smart Date (smart_date) fields consider all spans.
   */
  public function testMultiValueSmartDateField(): void {
    $multi = Node::create([
      'type' => 'booking',
      'title' => 'Multi',
      'uid' => 1,
      'field_smart_dates' => [
        [
          'value' => $this->dateStringToTimestamp('2024-07-04T09:00:00'),
          'end_value' => $this->dateStringToTimestamp('2024-07-04T10:00:00'),
        ],
        [
          'value' => $this->dateStringToTimestamp('2024-07-04T12:00:00'),
          'end_value' => $this->dateStringToTimestamp('2024-07-04T13:00:00'),
        ],
      ],
    ]);
    $multi->save();

    $other = $this->createSmartDateNode('Second slot conflict', '2024-07-04T12:15:00', '2024-07-04T12:45:00');
    $conflicts = $this->manager->getConflicts($other);

    $this->assertCount(1, $conflicts);
    $this->assertSame($multi->id(), reset($conflicts)->id());
  }

  /**
   * Ensures Smart Date recurring instances participate in detection.
   */
  public function testSmartDateRecurConflicts(): void {
    $rule = SmartDateRule::create([
      'rule' => 'FREQ=DAILY',
      'freq' => 'DAILY',
      'limit' => 'COUNT=2',
      'parameters' => '',
      'unlimited' => FALSE,
      'entity_type' => 'node',
      'bundle' => 'booking',
      'field_name' => 'field_smart_dates',
      'start' => $this->dateStringToTimestamp('2024-07-10T09:00:00'),
      'end' => $this->dateStringToTimestamp('2024-07-10T10:00:00'),
    ]);
    $rule->save();

    $instances = $rule->makeRuleInstances();
    $field_values = [];
    foreach ($instances as $delta => $instance) {
      $field_values[] = [
        'value' => $instance->getStart()->getTimestamp(),
        'end_value' => $instance->getEnd()->getTimestamp(),
        'rrule' => $rule->id(),
        'rrule_index' => $delta,
      ];
    }
    $this->assertCount(2, $field_values);

    $series = Node::create([
      'type' => 'booking',
      'title' => 'Series',
      'uid' => 1,
      'field_smart_dates' => $field_values,
    ]);
    $series->save();

    $other = $this->createSmartDateNode('Overlap recurring', '2024-07-11T09:30:00', '2024-07-11T10:30:00');
    $conflicts = $this->manager->getConflicts($other);

    $this->assertCount(1, $conflicts);
    $this->assertSame($series->id(), reset($conflicts)->id());
  }

  /**
   * Helper to create a Smart Date (smart_date) backed node.
   */
  protected function createSmartDateNode(string $title, string $start, string $end): Node {
    return Node::create([
      'type' => 'booking',
      'title' => $title,
      'uid' => 1,
      'field_smart_dates' => [
        [
          'value' => $this->dateStringToTimestamp($start),
          'end_value' => $this->dateStringToTimestamp($end),
        ],
      ],
    ]);
  }

  /**
   * Converts a date string to a UTC timestamp.
   */
  protected function dateStringToTimestamp(string $value): int {
    $date = new DrupalDateTime($value, 'UTC');
    return (int) $date->getTimestamp();
  }

}
