<?php

declare(strict_types=1);

namespace Drupal\Tests\resource_conflict\Functional;

use Drupal\Core\Field\FieldStorageDefinitionInterface;
use Drupal\node\Entity\Node;
use Drupal\node\Entity\NodeType;
use Drupal\resource_conflict\Service\ResourceConflictManager;
use Drupal\Tests\BrowserTestBase;

/**
 * Functional coverage for custom validation event subscribers.
 *
 * @group resource_conflict
 */
class ResourceConflictValidationSubscriberTest extends BrowserTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'node',
    'user',
    'field',
    'options',
    'datetime',
    'datetime_range',
    'resource_conflict',
    'resource_conflict_test_validation',
  ];

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * Resource Conflict manager.
   */
  protected ResourceConflictManager $manager;

  /**
   * Name of the test content type.
   */
  protected const string BUNDLE = 'rc_subscriber';

  /**
   * Machine name of the configured field.
   */
  protected const string FIELD_NAME = 'field_conflict_dates';

  /**
   * Machine name of the room reference field.
   */
  protected const string ROOM_FIELD_NAME = 'field_conflict_room';

  /**
   * Machine name of the room content type.
   */
  protected const string ROOM_BUNDLE = 'rc_room';

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Ensure state flag used by the test module starts clean.
    NodeType::create([
      'type' => static::BUNDLE,
      'name' => 'Resource conflict subscriber page',
    ])->save();

    NodeType::create([
      'type' => static::ROOM_BUNDLE,
      'name' => 'Room',
    ])->save();

    $storage = $this->container->get('entity_type.manager')->getStorage('field_storage_config');
    $storage->create([
      'entity_type' => 'node',
      'field_name' => static::FIELD_NAME,
      'type' => 'daterange',
      'cardinality' => FieldStorageDefinitionInterface::CARDINALITY_UNLIMITED,
    ])->save();

    $storage->create([
      'entity_type' => 'node',
      'field_name' => static::ROOM_FIELD_NAME,
      'type' => 'entity_reference',
      'cardinality' => 1,
      'settings' => [
        'target_type' => 'node',
      ],
    ])->save();

    $field_config = $this->container->get('entity_type.manager')->getStorage('field_config');
    $field_config->create([
      'entity_type' => 'node',
      'field_name' => static::FIELD_NAME,
      'bundle' => static::BUNDLE,
      'label' => 'Conflict dates',
      'required' => TRUE,
    ])->save();

    $field_config->create([
      'entity_type' => 'node',
      'field_name' => static::ROOM_FIELD_NAME,
      'bundle' => static::BUNDLE,
      'label' => 'Room',
      'required' => FALSE,
      'settings' => [
        'handler' => 'default:node',
        'handler_settings' => [
          'target_bundles' => [static::ROOM_BUNDLE => static::ROOM_BUNDLE],
        ],
      ],
    ])->save();

    // Make newly created fields available to form/builders immediately.
    $this->container->get('entity_field.manager')->clearCachedFieldDefinitions();
    $this->container->get('entity_type.manager')->clearCachedDefinitions();

    $this->config('system.date')->set('timezone.default', 'UTC')->save();

    $this->manager = $this->container->get('resource_conflict.manager');

    $form_display = $this->container->get('entity_display.repository')->getFormDisplay('node', static::BUNDLE, 'default');
    $form_display->setComponent(static::FIELD_NAME, [
      'type' => 'daterange_default',
    ]);
    $form_display->setComponent(static::ROOM_FIELD_NAME, [
      'type' => 'entity_reference_autocomplete',
    ]);
    $form_display->save();

    $permissions = [
      "create " . static::BUNDLE . " content",
      "edit any " . static::BUNDLE . " content",
    ];
    $account = $this->drupalCreateUser($permissions);
    $this->drupalLogin($account);
  }

  /**
   * Ensures a subscriber can replace the default validation messaging.
   */
  public function testSubscriberProvidesValidationError(): void {
    $this->enableResourceConflict([
      'default_form_error' => FALSE,
    ]);

    $this->createNodeWithDates('Existing', '2024-10-01T09:00:00', '2024-10-01T10:00:00');

    $edit = [
      'title[0][value]' => 'Subscriber validation attempt',
    ] + $this->buildDateRangeInput('2024-10-01T09:30:00', '2024-10-01T10:30:00');

    $this->drupalGet('node/add/' . static::BUNDLE);
    $this->submitForm($edit, 'Save');

    $session = $this->assertSession();
    $session->statusMessageContains('Subscriber blocked submission for conflicting dates.', 'error');
    $session->pageTextNotContains('This entry conflicts with the following content');

    $matches = $this->container->get('entity_type.manager')->getStorage('node')->loadByProperties([
      'type' => static::BUNDLE,
      'title' => 'Subscriber validation attempt',
    ]);
    $this->assertCount(0, $matches);
  }

  /**
   * Ensures a subscriber can filter conflicts by an additional field.
   */
  public function testSubscriberFiltersConflictsByRoom(): void {
    $room_a = $this->createRoom('Main hall');
    $room_b = $this->createRoom('Side room');

    $this->createNodeWithDates('Room A original', '2024-10-05T09:00:00', '2024-10-05T10:00:00', $room_a);
    $this->createNodeWithDates('Room B original', '2024-10-05T09:00:00', '2024-10-05T10:00:00', $room_b);

    // Enable conflict checking after fixtures are created to avoid
    // pre-save blocks.
    $this->enableResourceConflict([
      'default_form_error' => FALSE,
    ]);

    // Refresh form display cache so the newly added room field appears.
    \Drupal::entityTypeManager()->getStorage('entity_form_display')->resetCache();

    $edit = [
      'title[0][value]' => 'Room-filtered attempt',
    ] + $this->buildDateRangeInput('2024-10-05T09:30:00', '2024-10-05T10:30:00') + $this->buildRoomInput($room_a);

    $this->drupalGet('node/add/' . static::BUNDLE);
    $this->submitForm($edit, 'Save');

    $session = $this->assertSession();
    $session->statusMessageContains('Room-level conflict with: Room A original', 'error');
    $session->pageTextNotContains('Room B original');

    $matches = $this->container->get('entity_type.manager')->getStorage('node')->loadByProperties([
      'type' => static::BUNDLE,
      'title' => 'Room-filtered attempt',
    ]);
    $this->assertCount(0, $matches);
  }

  /**
   * Conflicts on other rooms should not prevent saving.
   */
  public function testConflictsFilteredAwayAllowSave(): void {
    $room_a = $this->createRoom('Main hall');
    $room_b = $this->createRoom('Side room');

    $this->createNodeWithDates('Room B original', '2024-11-05T09:00:00', '2024-11-05T10:00:00', $room_b);

    // Enable conflict checking after fixtures are created to avoid
    // pre-save blocks.
    $this->enableResourceConflict([
      'default_form_error' => FALSE,
    ]);
    \Drupal::entityTypeManager()->getStorage('entity_form_display')->resetCache();

    $edit = [
      'title[0][value]' => 'Room A booking',
    ] + $this->buildDateRangeInput('2024-11-05T09:30:00', '2024-11-05T10:30:00') + $this->buildRoomInput($room_a);

    $this->drupalGet('node/add/' . static::BUNDLE);
    $this->submitForm($edit, 'Save');

    $session = $this->assertSession();
    $session->statusMessageContains('has been created.');

    $matches = $this->container->get('entity_type.manager')->getStorage('node')->loadByProperties([
      'type' => static::BUNDLE,
      'title' => 'Room A booking',
    ]);
    $this->assertCount(1, $matches);
  }

  /**
   * Date conflicts with no room selected should still block saving.
   */
  public function testNoRoomSelectedStillBlocks(): void {

    $this->createNodeWithDates('Existing', '2024-12-01T09:00:00', '2024-12-01T10:00:00');

    $this->enableResourceConflict([
      'default_form_error' => TRUE,
    ]);
    \Drupal::entityTypeManager()->getStorage('entity_form_display')->resetCache();

    $edit = [
      'title[0][value]' => 'No room selected',
    ] + $this->buildDateRangeInput('2024-12-01T09:30:00', '2024-12-01T10:30:00');

    $this->drupalGet('node/add/' . static::BUNDLE);
    $this->submitForm($edit, 'Save');

    $session = $this->assertSession();
    $session->statusMessageContains('Subscriber blocked submission for conflicting dates.', 'error');

    $matches = $this->container->get('entity_type.manager')->getStorage('node')->loadByProperties([
      'type' => static::BUNDLE,
      'title' => 'No room selected',
    ]);
    $this->assertCount(0, $matches);
  }

  /**
   * Multiple date ranges should all be filtered by room.
   */
  public function testMultipleDateRangesFilterByRoom(): void {
    $room_a = $this->createRoom('Main hall');
    $room_b = $this->createRoom('Side room');

    $this->createNodeWithDates('Room A morning', '2025-01-05T09:00:00', '2025-01-05T10:00:00', $room_a);
    $this->createNodeWithDates('Room B afternoon', '2025-01-05T15:00:00', '2025-01-05T16:00:00', $room_b);

    $this->enableResourceConflict([
      'default_form_error' => FALSE,
    ]);
    \Drupal::entityTypeManager()->getStorage('entity_form_display')->resetCache();

    $this->drupalGet('node/add/' . static::BUNDLE);
    // First submit adds a second widget while keeping required values intact.
    $first = [
      'title[0][value]' => 'Two slots same room',
      static::FIELD_NAME . '[0][value][date]' => '2025-01-05',
      static::FIELD_NAME . '[0][value][time]' => '09:30:00',
      static::FIELD_NAME . '[0][end_value][date]' => '2025-01-05',
      static::FIELD_NAME . '[0][end_value][time]' => '10:30:00',
    ] + $this->buildRoomInput($room_a);
    $this->submitForm($first, 'Add another item');

    $edit = $first + [
      static::FIELD_NAME . '[1][value][date]' => '2025-01-05',
      static::FIELD_NAME . '[1][value][time]' => '15:30:00',
      static::FIELD_NAME . '[1][end_value][date]' => '2025-01-05',
      static::FIELD_NAME . '[1][end_value][time]' => '16:30:00',
    ];

    $this->submitForm($edit, 'Save');

    $session = $this->assertSession();
    $session->statusMessageContains('Room-level conflict with: Room A morning', 'error');
    $session->pageTextNotContains('Room B afternoon');

    $matches = $this->container->get('entity_type.manager')->getStorage('node')->loadByProperties([
      'type' => static::BUNDLE,
      'title' => 'Two slots same room',
    ]);
    $this->assertCount(0, $matches);
  }

  /**
   * Editing an existing booking should still respect room-filtered conflicts.
   */
  public function testEditExistingRespectsRoomFilter(): void {
    $room_a = $this->createRoom('Main hall');
    $room_b = $this->createRoom('Side room');

    $existing = $this->createNodeWithDates('Existing Room A', '2025-02-10T09:00:00', '2025-02-10T10:00:00', $room_a);
    $this->createNodeWithDates('Other Room A', '2025-02-10T09:15:00', '2025-02-10T10:15:00', $room_a);
    $this->createNodeWithDates('Existing Room B', '2025-02-10T09:00:00', '2025-02-10T10:00:00', $room_b);

    $this->enableResourceConflict([
      'default_form_error' => FALSE,
    ]);
    \Drupal::entityTypeManager()->getStorage('entity_form_display')->resetCache();

    $edit = [
      'title[0][value]' => 'Existing Room A',
    ] + $this->buildDateRangeInput('2025-02-10T09:30:00', '2025-02-10T10:30:00') + $this->buildRoomInput($room_a);

    $this->drupalGet('node/' . $existing->id() . '/edit');
    $this->submitForm($edit, 'Save');

    $session = $this->assertSession();
    $session->statusMessageContains('Room-level conflict with: Other Room A', 'error');
    $session->pageTextNotContains('Existing Room B');
  }

  /**
   * Creates a test node with the configured date range.
   */
  protected function createNodeWithDates(string $title, string $start, string $end, ?Node $room = NULL): Node {
    $values = [
      'type' => static::BUNDLE,
      'title' => $title,
      static::FIELD_NAME => [
        'value' => $start,
        'end_value' => $end,
      ],
    ];
    if ($room) {
      $values[static::ROOM_FIELD_NAME] = [
        'target_id' => $room->id(),
      ];
    }
    $node = Node::create($values);
    $node->save();
    return $node;
  }

  /**
   * Creates a room node for use in conflict filtering.
   */
  protected function createRoom(string $name): Node {
    $room = Node::create([
      'type' => static::ROOM_BUNDLE,
      'title' => $name,
    ]);
    $room->save();
    return $room;
  }

  /**
   * Builds form values for the date range widget.
   */
  protected function buildDateRangeInput(string $start, string $end): array {
    [$start_date, $start_time] = explode('T', $start);
    [$end_date, $end_time] = explode('T', $end);
    return [
      static::FIELD_NAME . '[0][value][date]' => $start_date,
      static::FIELD_NAME . '[0][value][time]' => $start_time,
      static::FIELD_NAME . '[0][end_value][date]' => $end_date,
      static::FIELD_NAME . '[0][end_value][time]' => $end_time,
    ];
  }

  /**
   * Builds form values for the room selector.
   */
  protected function buildRoomInput(Node $room): array {
    return [
      static::ROOM_FIELD_NAME . '[0][target_id]' => $room->label() . ' (' . $room->id() . ')',
    ];
  }

  /**
   * Enables resource conflict checking for the test bundle.
   */
  protected function enableResourceConflict(array $overrides = []): void {
    $settings = [
      'enabled' => TRUE,
      'field_name' => static::FIELD_NAME,
      'restrict_to_bundle' => FALSE,
      'start_buffer' => '',
      'end_buffer' => '',
      'default_form_error' => TRUE,
    ];
    $this->manager->saveBundleSettings(static::BUNDLE, $overrides + $settings);
  }

}
