<?php

declare(strict_types=1);

namespace Drupal\Tests\resource_conflict\Kernel;

use Drupal\Core\Entity\EntityStorageException;
use Drupal\Core\Form\FormState;
use Drupal\Core\Field\FieldStorageDefinitionInterface;
use Drupal\field\Entity\FieldConfig;
use Drupal\field\Entity\FieldStorageConfig;
use Drupal\node\Entity\Node;
use Drupal\node\Entity\NodeType;
use Drupal\resource_conflict\Service\ResourceConflictManager;
use Drupal\resource_conflict_test\EventSubscriber\ResourceConflictTestSubscriber;
use Drupal\KernelTests\KernelTestBase;
use Drupal\user\Entity\User;

/**
 * Tests the Resource Conflict manager service.
 *
 * @group resource_conflict
 */
class ResourceConflictManagerTest extends KernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'user',
    'field',
    'text',
    'node',
    'datetime',
    'datetime_range',
    'resource_conflict',
    'resource_conflict_test',
  ];

  /**
   * The manager under test.
   */
  protected ResourceConflictManager $manager;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installEntitySchema('user');
    $this->installEntitySchema('node');
    $this->installSchema('system', ['sequences']);
    $this->installConfig(['node', 'resource_conflict']);

    // Create a user so nodes can reference uid 1.
    User::create([
      'name' => 'admin',
    ])->save();

    // Create the content type used for testing.
    NodeType::create([
      'type' => 'booking',
      'name' => 'Booking',
    ])->save();

    // Attach a daterange field.
    FieldStorageConfig::create([
      'entity_type' => 'node',
      'field_name' => 'field_booking_dates',
      'type' => 'daterange',
      'cardinality' => FieldStorageDefinitionInterface::CARDINALITY_UNLIMITED,
      'settings' => [
        'datetime_type' => 'datetime',
      ],
    ])->save();

    FieldConfig::create([
      'entity_type' => 'node',
      'field_name' => 'field_booking_dates',
      'bundle' => 'booking',
      'label' => 'Booking dates',
      'required' => FALSE,
    ])->save();

    $this->manager = $this->container->get('resource_conflict.manager');
    $this->manager->saveBundleSettings('booking', [
      'enabled' => TRUE,
      'field_name' => 'field_booking_dates',
      'restrict_to_bundle' => TRUE,
      'start_buffer' => '',
      'end_buffer' => '',
    ]);

    ResourceConflictTestSubscriber::reset();
  }

  /**
   * Ensures overlapping nodes trigger conflicts.
   */
  public function testConflictDetection(): void {
    $node_a = $this->createNodeWithDates('Booking A', '2024-01-01T10:00:00', '2024-01-01T11:00:00');
    $node_a->save();

    $node_b = $this->createNodeWithDates('Booking B', '2024-01-01T10:30:00', '2024-01-01T11:30:00');
    $conflicts = $this->manager->getConflicts($node_b);

    $this->assertCount(1, $conflicts);
    $this->assertSame($node_a->id(), reset($conflicts)->id());

    $this->expectException(EntityStorageException::class);
    $node_b->save();
  }

  /**
   * Ensures timezone differences do not hide overlaps.
   */
  public function testConflictDetectionWithLocalTimezone(): void {
    $original_timezone = date_default_timezone_get();
    date_default_timezone_set('America/Vancouver');

    $node_a = $this->createNodeWithDates('Local A', '2024-08-01T09:00:00', '2024-08-01T10:00:00');
    $node_a->save();

    $node_b = $this->createNodeWithDates('Local B', '2024-08-01T09:30:00', '2024-08-01T10:30:00');
    $conflicts = $this->manager->getConflicts($node_b);

    $this->assertCount(1, $conflicts);
    $this->assertSame($node_a->id(), reset($conflicts)->id());

    date_default_timezone_set($original_timezone);
  }

  /**
   * Ensures non-overlapping nodes do not conflict.
   */
  public function testNoConflictForSeparateTimes(): void {
    $node_a = $this->createNodeWithDates('Morning', '2024-01-01T08:00:00', '2024-01-01T09:00:00');
    $node_a->save();

    $node_b = $this->createNodeWithDates('Afternoon', '2024-01-01T10:00:00', '2024-01-01T11:00:00');
    $conflicts = $this->manager->getConflicts($node_b);

    $this->assertEmpty($conflicts);
    $node_b->save();
    $this->assertNotEmpty($node_b->id());
  }

  /**
   * Ensures buffers extend the conflict window.
   */
  public function testBuffersExtendTimeSpans(): void {
    $this->manager->saveBundleSettings('booking', [
      'enabled' => TRUE,
      'field_name' => 'field_booking_dates',
      'restrict_to_bundle' => TRUE,
      'start_buffer' => '-15 minutes',
      'end_buffer' => '+15 minutes',
    ]);

    $node_a = $this->createNodeWithDates('Early', '2024-03-01T08:00:00', '2024-03-01T09:00:00');
    $node_a->save();

    // Normally this node would start after node A ends, but the -15 minute
    // buffer causes the overlap.
    $node_b = $this->createNodeWithDates('Buffered overlap', '2024-03-01T09:10:00', '2024-03-01T10:00:00');

    $conflicts = $this->manager->getConflicts($node_b);
    $this->assertCount(1, $conflicts);
    $this->assertSame($node_a->id(), reset($conflicts)->id());
  }

  /**
   * Ensures multi-value fields consider every time span.
   */
  public function testMultiValueDateField(): void {
    $node_a = Node::create([
      'type' => 'booking',
      'title' => 'Multi A',
      'uid' => 1,
      'field_booking_dates' => [
        [
          'value' => '2024-04-01T09:00:00',
          'end_value' => '2024-04-01T10:00:00',
        ],
        [
          'value' => '2024-04-01T12:00:00',
          'end_value' => '2024-04-01T13:00:00',
        ],
      ],
    ]);
    $node_a->save();

    $node_b = $this->createNodeWithDates('Second slot conflict', '2024-04-01T12:30:00', '2024-04-01T13:30:00');
    $conflicts = $this->manager->getConflicts($node_b);

    $this->assertCount(1, $conflicts);
    $this->assertSame($node_a->id(), reset($conflicts)->id());
  }

  /**
   * Ensures bundle restriction limits conflict detection.
   */
  public function testRestrictToBundle(): void {
    NodeType::create([
      'type' => 'events',
      'name' => 'Events',
    ])->save();

    FieldConfig::create([
      'entity_type' => 'node',
      'field_name' => 'field_booking_dates',
      'bundle' => 'events',
      'label' => 'Event dates',
      'required' => TRUE,
    ])->save();

    $this->manager->saveBundleSettings('events', [
      'enabled' => TRUE,
      'field_name' => 'field_booking_dates',
      'restrict_to_bundle' => TRUE,
      'start_buffer' => '',
      'end_buffer' => '',
    ]);

    $booking = $this->createNodeWithDates('Booking', '2024-05-01T09:00:00', '2024-05-01T10:00:00');
    $booking->save();

    $event = Node::create([
      'type' => 'events',
      'title' => 'Event',
      'uid' => 1,
      'field_booking_dates' => [
        [
          'value' => '2024-05-01T09:30:00',
          'end_value' => '2024-05-01T10:30:00',
        ],
      ],
    ]);

    $conflicts = $this->manager->getConflicts($event);
    $this->assertEmpty($conflicts);
  }

  /**
   * Ensures events fire for validation and conflict detection.
   */
  public function testEventsDispatched(): void {
    $node_a = $this->createNodeWithDates('Original', '2024-02-01T09:00:00', '2024-02-01T10:00:00');
    $node_a->save();

    $node_b = $this->createNodeWithDates('Overlap', '2024-02-01T09:30:00', '2024-02-01T10:30:00');

    $form_state = new FormState();
    $form_object = \Drupal::entityTypeManager()->getFormObject('node', 'default');
    $form_object->setEntity($node_b);
    $form_state->setFormObject($form_object);
    $form_display = \Drupal::service('entity_display.repository')->getFormDisplay('node', 'booking', 'default');
    $form_state->set('form_display', $form_display);
    $form_state->setValue('field_booking_dates', [
      [
        'value' => '2024-02-01T09:30:00',
        'end_value' => '2024-02-01T10:30:00',
      ],
    ]);
    $form = $form_object->buildForm([], $form_state);
    resource_conflict_node_form_validate($form, $form_state);

    $this->assertSame(['Overlap'], ResourceConflictTestSubscriber::getValidationEvents());
  }

  /**
   * Ensures built-in validation can be disabled.
   */
  public function testDefaultFormErrorToggle(): void {
    $this->manager->saveBundleSettings('booking', [
      'enabled' => TRUE,
      'field_name' => 'field_booking_dates',
      'restrict_to_bundle' => TRUE,
      'start_buffer' => '',
      'end_buffer' => '',
      'default_form_error' => FALSE,
    ]);

    $node_a = $this->createNodeWithDates('Original', '2024-06-01T09:00:00', '2024-06-01T10:00:00');
    $node_a->save();

    $node_b = $this->createNodeWithDates('Other', '2024-06-01T09:30:00', '2024-06-01T10:30:00');

    $form_state = new FormState();
    $form_object = \Drupal::entityTypeManager()->getFormObject('node', 'default');
    $form_object->setEntity($node_b);
    $form_state->setFormObject($form_object);
    $form_display = \Drupal::service('entity_display.repository')->getFormDisplay('node', 'booking', 'default');
    $form_state->set('form_display', $form_display);
    $form_state->setValue('field_booking_dates', [
      [
        'value' => '2024-06-01T09:30:00',
        'end_value' => '2024-06-01T10:30:00',
      ],
    ]);
    $form = $form_object->buildForm([], $form_state);
    resource_conflict_node_form_validate($form, $form_state);

    $this->assertEmpty($form_state->getErrors());
  }

  /**
   * Helper to create a node with a date range value.
   */
  protected function createNodeWithDates(string $title, string $start, string $end): Node {
    return Node::create([
      'type' => 'booking',
      'title' => $title,
      'uid' => 1,
      'field_booking_dates' => [
        [
          'value' => $start,
          'end_value' => $end,
        ],
      ],
    ]);
  }

}
