<?php

declare(strict_types=1);

namespace Drupal\resource_conflict_test_validation\EventSubscriber;

use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\StringTranslation\TranslationInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\resource_conflict\Event\ConflictsEvent;
use Drupal\resource_conflict\Event\ConflictValidationEvent;
use Drupal\resource_conflict\Event\ResourceConflictEvents;
use Drupal\resource_conflict\Service\ResourceConflictManager;
use Drupal\node\NodeInterface;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

/**
 * Adds a custom validation error during Resource Conflict validation.
 */
class ResourceConflictValidationSubscriber implements EventSubscriberInterface {

  use StringTranslationTrait;

  /**
   * Room reference field used for filtering conflicts.
   */
  private const string ROOM_FIELD = 'field_conflict_room';

  /**
   * Constructs the subscriber.
   */
  public function __construct(
    protected ResourceConflictManager $manager,
    protected EntityTypeManagerInterface $entityTypeManager,
    TranslationInterface $stringTranslation,
  ) {
    $this->stringTranslation = $stringTranslation;
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    return [
      ResourceConflictEvents::CONFLICT_VALIDATION => 'onConflictValidation',
      ResourceConflictEvents::CONFLICTS_FILTER => 'onConflictsFilter',
    ];
  }

  /**
   * Adds a validation error when conflicts are detected.
   */
  public function onConflictValidation(ConflictValidationEvent $event): void {
    $settings = $this->manager->getBundleSettings($event->getNode()->bundle());
    $field_name = $settings['field_name'] ?? '';
    $node = $event->getNode();
    $filtered_conflicts = NULL;
    $conflicts = $event->getConflicts();
    if (!empty($conflicts)) {
      $filtered_conflicts = $this->filterConflictsByRoom($node, $conflicts);
      $conflicts = $filtered_conflicts ?? $conflicts;
    }

    if (!empty($conflicts)) {
      $message = $this->t('Subscriber blocked submission for conflicting dates.');

      if ($filtered_conflicts !== NULL) {
        $labels = array_map(static fn (NodeInterface $conflict) => $conflict->label(), $conflicts);
        $message = $this->t('Room-level conflict with: @list', ['@list' => implode(', ', $labels)]);
      }

      $form = $event->getForm();
      if ($field_name && isset($form[$field_name])) {
        $event->getFormState()->setError($form[$field_name], $message);
        return;
      }
      $event->getFormState()->setErrorByName('', $message);
    }
  }

  /**
   * Filters conflicts to those matching the selected room.
   */
  protected function filterConflictsByRoom(NodeInterface $node, array $conflicts): ?array {
    if (!$node->hasField(self::ROOM_FIELD) || $node->get(self::ROOM_FIELD)->isEmpty()) {
      return NULL;
    }

    $target_id = (int) $node->get(self::ROOM_FIELD)->target_id;
    return array_filter($conflicts, static function (NodeInterface $conflict) use ($target_id) {
      return $conflict->hasField(self::ROOM_FIELD)
        && !$conflict->get(self::ROOM_FIELD)->isEmpty()
        && (int) $conflict->get(self::ROOM_FIELD)->target_id === $target_id;
    });
  }

  /**
   * Filters conflicts via event before validation/presave uses them.
   */
  public function onConflictsFilter(ConflictsEvent $event): void {
    $conflict_ids = $event->getConflictIds();
    if (!$conflict_ids) {
      return;
    }

    $conflicts = $this->entityTypeManager->getStorage('node')->loadMultiple($conflict_ids);
    $filtered = $this->filterConflictsByRoom($event->getNode(), $conflicts);
    if ($filtered !== NULL) {
      $event->setConflictIds(array_keys($filtered));
    }
  }

}
