<?php

declare(strict_types=1);

namespace Drupal\Tests\resource_conflict\Kernel;

use Drupal\Core\Entity\EntityStorageException;
use Drupal\Core\Field\FieldStorageDefinitionInterface;
use Drupal\field\Entity\FieldConfig;
use Drupal\field\Entity\FieldStorageConfig;
use Drupal\KernelTests\KernelTestBase;
use Drupal\node\Entity\Node;
use Drupal\node\Entity\NodeType;
use Drupal\resource_conflict\Service\ResourceConflictManager;
use Drupal\user\Entity\User;

/**
 * Integration coverage for Recurring Dates Field support.
 *
 * @group resource_conflict
 */
class ResourceConflictDateRecurTest extends KernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'user',
    'field',
    'text',
    'node',
    'datetime',
    'datetime_range',
    'date_recur',
    'resource_conflict',
  ];

  /**
   * The manager under test.
   */
  protected ResourceConflictManager $manager;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installEntitySchema('user');
    $this->installEntitySchema('node');
    $this->installSchema('system', ['sequences']);
    $this->installConfig(['node', 'resource_conflict']);

    User::create(['name' => 'admin'])->save();

    NodeType::create([
      'type' => 'booking',
      'name' => 'Booking',
    ])->save();

    FieldStorageConfig::create([
      'entity_type' => 'node',
      'field_name' => 'field_booking_recurring',
      'type' => 'date_recur',
      'cardinality' => FieldStorageDefinitionInterface::CARDINALITY_UNLIMITED,
      'settings' => [
        'datetime_type' => 'datetime',
      ],
    ])->save();

    FieldConfig::create([
      'entity_type' => 'node',
      'field_name' => 'field_booking_recurring',
      'bundle' => 'booking',
      'label' => 'Recurring dates',
      'required' => FALSE,
    ])->save();

    $this->manager = $this->container->get('resource_conflict.manager');
  }

  /**
   * Ensures Recurring Dates fields are exposed as eligible options.
   */
  public function testDateRecurFieldListedAsEligible(): void {
    $options = $this->manager->getEligibleFieldOptions('booking');
    $this->assertArrayHasKey('field_booking_recurring', $options);
  }

  /**
   * Ensures recurring occurrences participate in conflict detection.
   */
  public function testRecurringDateConflicts(): void {
    $this->manager->saveBundleSettings('booking', [
      'enabled' => TRUE,
      'field_name' => 'field_booking_recurring',
      'restrict_to_bundle' => TRUE,
      'start_buffer' => '',
      'end_buffer' => '',
    ]);

    $series_a = $this->createRecurringNode('Series A', '2024-07-01T09:00:00', '2024-07-01T10:00:00', 'FREQ=DAILY;COUNT=3');
    $series_a->save();

    $series_b = $this->createRecurringNode('Series B', '2024-07-02T09:30:00', '2024-07-02T10:30:00', 'FREQ=DAILY;COUNT=2');
    $conflicts = $this->manager->getConflicts($series_b);

    $this->assertCount(1, $conflicts);
    $this->assertSame($series_a->id(), reset($conflicts)->id());

    $this->expectException(EntityStorageException::class);
    $series_b->save();
  }

  /**
   * Ensures recurring conflicts are detected when using a local timezone.
   *
   * This guards against regressions where occurrences were built in the site
   * timezone (e.g., America/Vancouver) while stored values are UTC, causing
   * comparisons to miss.
   */
  public function testRecurringDateConflictsWithLocalTimezone(): void {
    $this->manager->saveBundleSettings('booking', [
      'enabled' => TRUE,
      'field_name' => 'field_booking_recurring',
      'restrict_to_bundle' => TRUE,
      'start_buffer' => '',
      'end_buffer' => '',
    ]);

    $original_timezone = date_default_timezone_get();
    date_default_timezone_set('America/Vancouver');

    try {
      $series_a = $this->createRecurringNode('Series A', '2024-07-01T09:00:00', '2024-07-01T10:00:00', 'FREQ=DAILY;COUNT=2', 'America/Vancouver');
      $series_a->save();

      // B overlaps the second occurrence of A (2024-07-02 at 09:30 local).
      $series_b = $this->createRecurringNode('Series B', '2024-07-02T09:30:00', '2024-07-02T10:30:00', 'FREQ=DAILY;COUNT=1', 'America/Vancouver');
      $conflicts = $this->manager->getConflicts($series_b);

      $this->assertCount(1, $conflicts);
      $this->assertSame($series_a->id(), reset($conflicts)->id());
    }
    finally {
      date_default_timezone_set($original_timezone);
    }
  }

  /**
   * Helper to create a node with a Recurring Dates field value.
   */
  protected function createRecurringNode(string $title, string $start, string $end, string $rrule, string $timezone = 'UTC'): Node {
    return Node::create([
      'type' => 'booking',
      'title' => $title,
      'uid' => 1,
      'field_booking_recurring' => [
        [
          'value' => $start,
          'end_value' => $end,
          'timezone' => $timezone,
          'rrule' => $rrule,
        ],
      ],
    ]);
  }

}
