<?php

namespace Drupal\rest_easy_api_keys\Authentication;

use Drupal\Core\Authentication\AuthenticationProviderInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\rest_easy\Plugin\APIManager;
use Drupal\rest_easy\Plugin\EndpointBase;
use Symfony\Component\HttpFoundation\Request;

/**
 * A simple API key authentication provider.
 *
 * @package Drupal\rest_easy_api_keys\Authentication
 */
class RESTEasyAPIKeyAuth implements AuthenticationProviderInterface {

  /**
   * A list of paths for REST Easy endpoints using this authentication method.
   *
   * @var array
   */
  protected $paths = [];

  /**
   * Entity type manager service.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * Constructs a new APIKeyAuth object.
   *
   * @param \Drupal\rest_easy\Plugin\APIManager $api_manager
   *   REST Easy API manager service.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   Entity type manager service.
   */
  public function __construct(APIManager $api_manager, EntityTypeManagerInterface $entityTypeManager) {
    foreach ($api_manager->getDefinitions() as $api_id => $api_definition) {
      try {
        $api = $api_manager->createInstance($api_id);
        foreach ($api->routes() as $route) {
          $auth_plugins = $route->getOption('_auth');
          if ($auth_plugins && in_array('rest_easy_api_key', $auth_plugins)) {
            $path = $route->getPath();
            if (strpos($path, '{') !== FALSE) {
              $path = substr($path, 0, strpos($path, '{'));
            }
            $this->paths[] = $path;
          }
        }
      }
      catch (\Exception $exception) {
        continue;
      }
    }
    $this->entityTypeManager = $entityTypeManager;
  }

  /**
   * {@inheritdoc}
   */
  public function applies(Request $request) {

    // Check if the request is for an endpoint that uses this authentication
    // method.
    $in_paths = FALSE;
    $current_path = $request->getRequestUri();
    foreach ($this->paths as $path) {
      if (str_starts_with($current_path, $path)) {
        $in_paths = TRUE;
        break;
      }
    }
    if (!$in_paths) {
      return FALSE;
    }

    // Check if the request has an API key.
    return $request->query->has('apiKey') || $request->headers->has('x-api-key');
  }

  /**
   * {@inheritdoc}
   */
  public function authenticate(Request $request) {
    $api_key = $request->query->get('apiKey') ?: $request->headers->get('x-api-key');
    if (!$api_key) {
      EndpointBase::status(403);
    }
    $userStorage = $this->entityTypeManager->getStorage('user');
    $user = $userStorage->getQuery()
      ->accessCheck(FALSE)
      ->condition('rest_easy_api_keys', $api_key)
      ->execute();
    if (count($user) > 0) {
      $user = $userStorage->load(reset($user));
    }
    if (!$user) {
      EndpointBase::status(403);
    }
    return $user;
  }

}
