<?php

namespace Drupal\rest_easy\Controller;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Controller\ControllerBase;
use Drupal\rest_easy\Plugin\EndpointManager;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Resolver for calls to REST Easy endpoint plugins.
 *
 * @package Drupal\rest_easy\Routing
 */
class EndpointController extends ControllerBase {

  /**
   * Endpoint plugin manager service.
   *
   * @var \Drupal\rest_easy\Plugin\EndpointManager
   */
  protected $endpointManager;

  /**
   * Constructs a new EndpointController object.
   *
   * @param \Drupal\rest_easy\Plugin\EndpointManager $endpointManager
   *   Endpoint plugin manager service.
   */
  public function __construct(EndpointManager $endpointManager) {
    $this->endpointManager = $endpointManager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('plugin.manager.rest_easy.endpoint')
    );
  }

  /**
   * Load the endpoint plugin and check access.
   *
   * @param string $plugin_id
   *   The ID of the endpoint plugin to load.
   *
   * @return \Drupal\Core\Access\AccessResult
   *   Whether the user has access to call the endpoint.
   */
  public function access(string $plugin_id): AccessResult {
    $arguments = func_get_args();
    unset($arguments['plugin_id']);
    $endpoint = $this->endpointManager->createInstance($plugin_id);
    return $endpoint->access($arguments);
  }

  /**
   * Load and execute the endpoint plugin.
   *
   * @param string $plugin_id
   *   The ID of the endpoint plugin to execute.
   *
   * @return mixed
   *   The result of the endpoint plugin's call() method.
   */
  public function call(string $plugin_id): mixed {
    $arguments = func_get_args();
    unset($arguments['plugin_id']);
    $endpoint = $this->endpointManager->createInstance($plugin_id);
    return $endpoint->call($arguments);
  }

}
