<?php

namespace Drupal\rest_easy\Plugin;

use Drupal\Component\Plugin\PluginInspectionInterface;
use Drupal\Core\Access\AccessResult;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Route;

/**
 * An interface for creating REST Easy endpoint plugins.
 *
 * @package Drupal\rest_easy\Plugin
 */
interface EndpointInterface extends PluginInspectionInterface {

  /**
   * Determine if a user can access the endpoint.
   *
   * By default, this is true if a user has the permission specified in the
   * endpoint plugin annotation. If no permission is supplied, it will back up
   * to the permission supplied by the API plugin annotation. If neither
   * specify a permission, any request will be allowed. Override this method
   * for endpoints with more sophisticated access handling.
   *
   * @param array $arguments
   *   An associative array of path arguments.
   *
   * @return \Drupal\Core\Access\AccessResult
   *   Whether the user can access the endpoint.
   */
  public function access(...$arguments): AccessResult;

  /**
   * Execute the endpoint and return the appropriate response.
   *
   * @return mixed
   *   The endpoint's response.
   */
  public function call(...$arguments): mixed;

  /**
   * Generate HTTP headers for the endpoint's responses.
   *
   * @return array
   *   An associative array of HTTP headers
   */
  public function headers(): array;

  /**
   * Package the data into a JSON response.
   *
   * @param mixed $data
   *   The data to include in the response.
   *
   * @return \Symfony\Component\HttpFoundation\Response
   *   The JSON response.
   */
  public function jsonResponse($data): Response;

  /**
   * Generate path data for OpenAPI documentation.
   *
   * @return array
   *   An array containing structured path information.
   */
  public function path(): array;

  /**
   * Generate the route for the endpoint.
   *
   * @return \Symfony\Component\Routing\Route
   *   The route object.
   *
   * @see \Drupal\rest_easy\Plugin\APIBase::routes()
   * @see \Drupal\rest_easy\Routing\APIRouteBuilder::routes()
   */
  public function route(): Route;

  /**
   * Generate a list of schema definitions for the OpenAPI documentation.
   *
   * @return array
   *   An associative array of schema definitions
   *
   * @see https://swagger.io/specification/v2/#definitions-object-example
   */
  public function schemaDefinitions(): array;

}
