<?php

namespace Drupal\rest_easy\Plugin;

use Drupal\Component\Plugin\PluginBase;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\rest_easy\Event\ParameterGetEvent;
use Drupal\rest_easy\Event\ParameterValidateEvent;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * A base class for building parameter plugins.
 *
 * @package Drupal\rest_easy\Plugin
 */
abstract class ParameterBase extends PluginBase implements ContainerFactoryPluginInterface, ParameterInterface {

  use StringTranslationTrait;

  /**
   * Event dispatcher service.
   *
   * @var \Symfony\Component\EventDispatcher\EventDispatcherInterface
   */
  public EventDispatcherInterface $eventDispatcher;

  /**
   * Current request.
   *
   * @var \Symfony\Component\HttpFoundation\Request
   */
  protected Request $request;

  /**
   * Route match service.
   *
   * @var \Drupal\Core\Routing\RouteMatchInterface
   */
  protected RouteMatchInterface $routeMatch;

  /**
   * Constructs a new ParameterPluginBase object.
   *
   * @param array $configuration
   *   Plugin configuration.
   * @param mixed $plugin_id
   *   Plugin ID.
   * @param mixed $plugin_definition
   *   Plugin definition.
   * @param \Symfony\Component\EventDispatcher\EventDispatcherInterface $eventDispatcher
   *   Event dispatcher service.
   * @param \Symfony\Component\HttpFoundation\RequestStack $request_stack
   *   Request stack service.
   * @param \Drupal\Core\Routing\RouteMatchInterface $routeMatch
   *   Route match service.
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, EventDispatcherInterface $eventDispatcher, RequestStack $request_stack, RouteMatchInterface $routeMatch) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->eventDispatcher = $eventDispatcher;
    $this->request = $request_stack->getCurrentRequest();
    $this->routeMatch = $routeMatch;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('event_dispatcher'),
      $container->get('request_stack'),
      $container->get('current_route_match')
    );
  }

  /**
   * Retrieve the value of the parameter from the request.
   *
   * @return mixed
   *   The value of the parameter.
   */
  public function get(): mixed {

    // Retrieve the value from the appropriate part of the request.
    $definition = $this->getPluginDefinition();
    $value = NULL;
    switch ($definition['in']) {
      case 'body':
        $body = &drupal_static('rest_easy_request_body');
        if (is_null($body)) {
          $body = $this->request->getContent();
          $body = json_decode($body) ?? new \stdClass();
        }
        $value = $body->{$definition['id']} ?? NULL;
        break;

      case 'formData':
        $value = $this->request->request->get($definition['id']);
        break;

      case 'header':
        $value = $this->request->headers->get($definition['id']);
        break;

      case 'path':
        $value = $this->routeMatch->getParameter($definition['id']);
        break;

      case 'query':
        $value = $this->request->query->get($definition['id']);
        break;
    }

    // Ignore empty parameters.
    if (is_string($value) && strlen($value) == 0) {
      $value = NULL;
    }

    // If no value is provided, set it to the parameter's default.
    if ($definition['default'] && is_null($value)) {
      $value = $definition['default'];
    }

    // Cast integer or string values as booleans.
    if ($definition['type'] == 'boolean' && $value !== NULL) {
      if ($value === 1 || $value === '1' || strtolower(trim($value)) == 'true') {
        $value = TRUE;
      }
      elseif ($value === 0 || $value === '0' || strtolower(trim($value)) == 'false') {
        $value = FALSE;
      }
    }

    // Cast strings as integers.
    if ($definition['type'] == 'integer' && is_string($value) && (int) $value == $value) {
      $value = (int) $value;
    }

    // Dispatch an event to allow other modules to alter the value.
    $event = new ParameterGetEvent($this, $value);
    $this->eventDispatcher->dispatch($event, ParameterGetEvent::EVENT_NAME);
    return $event->value;
  }

  /**
   * Validate the parameter value.
   *
   * @param mixed $value
   *   The value of the parameter.
   *
   * @return array
   *   An array of strings describing validation errors. If this is empty,
   *   validation was successful.
   */
  public function validate($value): array {
    $definition = $this->getPluginDefinition();
    $errors = [];

    // Handle null values.
    if (is_null($value)) {

      // Verify that required parameters are non-null.
      if ($definition['required']) {
        $errors[] = $this->t('The :parameter parameter is required.', [':parameter' => $definition['id']])
          ->render();
      }

      // Return immediately, because no further validation can take place.
      return $errors;
    }

    // Verify that enumerated parameters match one of their allowed values.
    if ($definition['enum'] && !in_array($value, $definition['enum'])) {
      $errors[] = $this->t('The :parameter parameter must be one of the following values: :allowed_values', [
        ':parameter' => $definition['id'],
        ':allowed_values' => implode(', ', $definition['enum']),
      ])->render();
    }

    // Validate by parameter type.
    switch ($definition['type']) {

      // Validate array parameters.
      case 'array':
        if (!is_array($value)) {
          $errors[] = $this->t('The :parameter parameter must be an array.', [':parameter' => $definition['id']])
            ->render();
        }

        // Verify that array parameters do not exceed their maximum number of
        // items.
        elseif (isset($definition['maxItems']) && count($value) > $definition['maxItems']) {
          $errors[] = $this->t('The :parameter parameter cannot have more than :maximum items.', [
            ':parameter' => $definition['id'],
            ':maximum' => $definition['maxItems'],
          ])->render();
        }
        break;

      // Validate boolean parameters.
      case 'boolean':
        if (!is_bool($value)) {
          $errors[] = $this->t('The :parameter parameter must be boolean.', [':parameter' => $definition['id']])
            ->render();
        }
        break;

      // Validate numeric parameters.
      case 'integer':
      case 'number':
        $numeric = TRUE;
        if ($definition['type'] == 'integer' && !is_int($value)) {
          $errors[] = $this->t('The :parameter parameter must be an integer.', [':parameter' => $definition['id']])
            ->render();
          $numeric = FALSE;
        }
        if ($definition['type'] == 'number' && !is_numeric($value)) {
          $errors[] = $this->t('The :parameter parameter must be a number.', [':parameter' => $definition['id']]);
          $numeric = FALSE;
        }

        // Verify that numeric parameters do not exceed their maximum value.
        if ($numeric) {
          if (isset($definition['maximum'])) {
            if ($definition['exclusiveMaximum'] && $value >= $definition['maximum']) {
              $errors[] = $this->t('The :parameter parameter must less than :maximum.', [
                ':parameter' => $definition['id'],
                ':maximum' => $definition['maximum'],
              ])->render();
            }
            elseif ($value > $definition['maximum']) {
              $errors[] = $this->t('The :parameter parameter must less than or equal to :maximum.', [
                ':parameter' => $definition['id'],
                ':maximum' => $definition['maximum'],
              ])->render();
            }
          }

          // Verify that numeric parameters do not fall below their minimum
          // value.
          if (isset($definition['minimum'])) {
            if ($definition['exclusiveMinimum'] && $value <= $definition['minimum']) {
              $errors[] = $this->t('The :parameter parameter must greater than :minimum.', [
                ':parameter' => $definition['id'],
                ':minimum' => $definition['minimum'],
              ])->render();
            }
            elseif ($value < $definition['minimum']) {
              $errors[] = $this->t('The :parameter parameter must less than or equal to :minimum.', [
                ':parameter' => $definition['id'],
                ':minimum' => $definition['minimum'],
              ])->render();
            }
          }
        }
        break;

      // Validate string parameters.
      case 'string':
        if (!is_string($value)) {
          $errors[] = $this->t('The :parameter parameter must be a string.', [':parameter' => $definition['id']])
            ->render();
        }
        else {

          // Verify that string parameters do not exceed their maximum string
          // length.
          if (isset($definition['maxLength']) && strlen($value) > $definition['maxLength']) {
            $errors[] = $this->t('The :parameter parameter cannot be longer than :maximum characters in length.', [
              ':parameter' => $definition['id'],
              ':maximum' => $definition['maxLength'],
            ])->render();
          }

          // Verify that string parameters do not fall below their minimum
          // string length.
          if (isset($definition['minLength']) && strlen($value) < $definition['minLength']) {
            $errors[] = $this->t('The :parameter parameter cannot be shorter than :minimum characters in length.', [
              ':parameter' => $definition['id'],
              ':minimum' => $definition['minLength'],
            ])->render();
          }

          // Verify that string parameters match their required regex pattern.
          if (isset($definition['pattern']) && !preg_match('/' . $definition['pattern'] . '/', $value)) {
            $errors[] = $this->t('The :parameter parameter must match the pattern: :pattern', [
              ':parameter' => $definition['id'],
              ':pattern' => $definition['pattern'],
            ])->render();
          }
        }
        break;
    }

    // Dispatch an event to allow other modules to alter validation.
    $event = new ParameterValidateEvent($this, $errors);
    $this->eventDispatcher->dispatch($event, ParameterValidateEvent::EVENT_NAME);
    return $event->errors;
  }

}
