<?php

namespace Drupal\rest_easy\Traits;

/**
 * Paginated endpoints include features to simplify paginated output.
 *
 * @package Drupal\rest_easy\Trait
 */
trait PaginatedEndpointTrait {

  /**
   * The plugin ID for an integer parameter.
   *
   * This is used to define the maximum number of results per page. It will
   * default to "limit" unless otherwise specified.
   *
   * @var string
   */
  protected $limitParameter;

  /**
   * The plugin ID for an integer parameter.
   *
   * This is used to determine how many results to skip prior to the current
   * page. It will default to "offset" unless otherwise specified.
   *
   * @var string
   */
  protected $offsetParameter;

  /**
   * The schema for the paging portion of a response.
   *
   * @var array
   */
  protected $pagingSchema = [
    'Paging' => [
      'description' => 'Pagination information used to traverse result sets',
      'properties' => [
        'count' => [
          'description' => 'The number of records in the current response',
          'type' => 'integer',
        ],
        'currentPage' => [
          'description' => 'The current page number',
          'type' => 'integer',
        ],
        'hasNext' => [
          'description' => 'Whether or not there are more records available in the current result set at a higher offset',
          'type' => 'boolean',
        ],
        'hasPrev' => [
          'description' => 'Whether or not there are more records available in the current result set at a lower offset',
          'type' => 'boolean',
        ],
        'limit' => [
          'description' => 'The maximum number of records in the current response. If this is set to 0, all records in the available result set are returned',
          'type' => 'integer',
        ],
        'offset' => [
          'description' => 'The number of records in the available result set that were skipped',
          'type' => 'integer',
        ],
        'nextOffset' => [
          'description' => 'The offset for the next page of results, or boolean false if not applicable',
          'type' => 'integer',
        ],
        'nextPath' => [
          'description' => 'The URL path for the next page of results, or boolean false if not applicable',
          'format' => 'url',
          'type' => 'string',
        ],
        'pageCount' => [
          'description' => 'The total number of pages available in the current result set',
          'type' => 'integer',
        ],
        'prevOffset' => [
          'description' => 'The offset for the previous page of results, or boolean false if not applicable',
          'type' => 'integer',
        ],
        'prevPath' => [
          'description' => 'The URL path for the previous page of results, or boolean false if not applicable',
          'format' => 'url',
          'type' => 'string',
        ],
        'totalCount' => [
          'description' => 'The total number of records available in the current result set',
          'type' => 'integer',
        ],
      ],
      'type' => 'object',
    ],
  ];

  /**
   * Generate the paging portion of a response.
   *
   * @param int $result_count
   *   The number of results in the current response.
   * @param int $total_count
   *   The total number of results available via paging.
   *
   * @return array
   *   An associative array of paging information.
   */
  public function paging(int $result_count, int $total_count): array {
    $limit_parameter = $this->limitParameter ?? 'limit';
    $limit = $this->parameters[$limit_parameter];
    $offset_parameter = $this->offsetParameter ?? 'offset';
    $offset = $this->parameters[$offset_parameter];
    $paging = [
      'count' => $result_count,
      'currentPage' => $limit == 0 ? 1 : ceil($offset / $limit) + 1,
      'hasNext' => $limit != 0 && $result_count + $offset < $total_count,
      'hasPrev' => $limit != 0 && $offset != 0,
      'limit' => $limit,
      'offset' => $offset,
      'nextOffset' => FALSE,
      'nextPath' => FALSE,
      'pageCount' => $limit == 0 ? 1 : ceil($total_count / $limit),
      'prevOffset' => FALSE,
      'prevPath' => FALSE,
      'totalCount' => $total_count,
    ];
    if ($paging['hasNext']) {
      $paging['nextOffset'] = $offset + $limit;
      $next_parameters = $this->request->query->all();
      $next_parameters[$offset_parameter] = $paging['nextOffset'];
      $path = explode('?', $this->request->getRequestUri());
      $paging['nextPath'] = $path[0] . '?' . http_build_query($next_parameters);
    }
    if ($paging['hasPrev']) {
      $paging['prevOffset'] = max(0, $offset - $limit);
      $prev_parameters = $this->request->query->all();
      if ($paging['prevOffset']) {
        $prev_parameters[$offset_parameter] = $paging['prevOffset'];
      }
      else {
        unset($prev_parameters['offset']);
      }
      $path = explode('?', $this->request->getRequestUri());
      $paging['prevPath'] = $path[0] . '?' . http_build_query($prev_parameters);
    }
    return $paging;
  }

  /**
   * Set the limit parameter used by the endpoint.
   *
   * @param string $parameter
   *   The plugin ID for a integer parameter used to establish the maximum
   *   number of results per page.
   */
  protected function setLimitParameter(string $parameter): void {
    $this->limitParameter = $parameter;
  }

  /**
   * Set the offset parameter used by the endpoint.
   *
   * @param string $parameter
   *   The plugin ID for a integer parameter used to determine how many results
   *   to skip prior to the current page.
   */
  protected function setOffsetParameter(string $parameter): void {
    $this->offsetParameter = $parameter;
  }

}
