<?php

namespace Drupal\rest_easy_basic_auth\EventSubscriber;

use Drupal\rest_easy\Event\APIRoutesEvent;
use Drupal\rest_easy\Event\APISecurityDefinitionsEvent;
use Drupal\rest_easy\Event\APISecurityEvent;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

/**
 * An event subscriber to alter OpenAPI security definitions and entries.
 *
 * @package Drupal\rest_easy_basic_auth\EventSubscriber
 */
class APIEventSubscriber implements EventSubscriberInterface {

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents() {
    return [
      APIRoutesEvent::EVENT_NAME => 'onRoutes',
      APISecurityEvent::EVENT_NAME => 'onSecurity',
      APISecurityDefinitionsEvent::EVENT_NAME => 'onSecurityDefinitions',
    ];
  }

  /**
   * React to the routes event.
   *
   * @param \Drupal\rest_easy\Event\APIRoutesEvent $event
   *   The event object.
   */
  public function onRoutes(APIRoutesEvent $event) {
    foreach ($event->routes as &$route) {
      $auth = $route->getOption('_auth');
      if (in_array('basic_auth', $auth)) {
        $requirements = $route->getRequirements();
        $requirements['_user_is_logged_in'] = 'TRUE';
        $route->setRequirements($requirements);
      }
    }
  }

  /**
   * React to the security event.
   *
   * @param \Drupal\rest_easy\Event\APISecurityEvent $event
   *   The event object.
   */
  public function onSecurity(APISecurityEvent $event) {

    // Check if the API uses the Basic Auth authentication method.
    $definition = $event->api->getPluginDefinition();
    if (isset($definition['auth'][0]) && in_array('basic_auth', $definition['auth'])) {
      $event->security[] = ['basicAuth' => []];
    }
  }

  /**
   * React to the security definitions event.
   *
   * @param \Drupal\rest_easy\Event\APISecurityDefinitionsEvent $event
   *   The event object.
   */
  public function onSecurityDefinitions(APISecurityDefinitionsEvent $event) {

    // Check if the API uses the Basic Auth authentication method.
    $definition = $event->api->getPluginDefinition();
    if (isset($definition['auth'][0]) && in_array('basic_auth', $definition['auth'])) {
      $event->definitions['basicAuth'] = [
        'type' => 'basic',
      ];
    }
  }

}
