<?php

namespace Drupal\rest_easy_example\Plugin\rest_easy\Endpoint;

use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\rest_easy\Attribute\RESTEasyEndpoint;
use Drupal\rest_easy\Plugin\EndpointBase;
use Drupal\rest_easy\Traits\CacheableEndpointTrait;
use Symfony\Component\HttpFoundation\Response;

/**
 * An example REST Easy endpoint plugin.
 *
 * @package Drupal\rest_easy_example\Plugin\rest_easy\Endpoint
 */
#[RESTEasyEndpoint(
  api: "rest_easy_example",
  id: "site_name",
  label: new TranslatableMarkup("Site Name Endpoint"),
  path: "/site-name",
  auth: ["cookie"],
  description: new TranslatableMarkup("A simple endpoint to get the site name."),
  methods: ["GET"],
  parameters: ["pretty"],
  responses: [200, 400],
  tags: ["Example"]
)]
class SiteNameEndpoint extends EndpointBase {

  // Adding CacheableEndpointTrait to the class gives us access to cacheGet()
  // and cacheSet(), which automatically vary cache IDs based on the
  // endpoint's arguments and parameters.
  use CacheableEndpointTrait;

  /**
   * {@inheritdoc}
   */
  public function call(...$arguments): mixed {

    // Every endpoint plugin must implement a call() method that executes the
    // endpoint's logic and returns an appropriate response. The $arguments
    // array will contain any parameters in the path (e.g., "/endpoint/{node}"
    // will pass $arguments['node']). Other parameters will be available in the
    // $this->parameters array. The jsonResponse() method can be used to return
    // a JSON response.
    if (!($data = $this->cacheGet())) {
      $data = [
        'site_name' => \Drupal::config('system.site')->get('name'),
      ];
      $this->cacheSet($data);
    }

    return $this->jsonResponse($data);
  }

  /**
   * {@inheritdoc}
   */
  public function jsonResponse($data): Response {

    // Parameters listed in the plugin attributes are automatically validated
    // and parsed into the parameters array. Here, we're checking if the
    // "pretty" parameter is true and, if so, setting the JSON encoding options
    // to include pretty-printing.
    $response = parent::jsonResponse($data);
    if ($this->parameters['pretty']) {
      $response->setContent(json_encode($data, JSON_PRETTY_PRINT));
    }
    return $response;
  }

}
