<?php

namespace Drupal\rest_easy_postman\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Link;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\Url;
use Drupal\rest_easy\Plugin\APIManager;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\RequestStack;
use Symfony\Component\HttpFoundation\Response;

/**
 * Lists downloadable Postman collections for REST Easy APIs.
 *
 * @package Drupal\rest_easy_postman\Routing
 */
class PostmanCollectionController extends ControllerBase {
  use StringTranslationTrait;

  /**
   * API plugin manager service.
   *
   * @var \Drupal\rest_easy\Plugin\APIManager
   */
  protected $apiManager;

  /**
   * Current request.
   *
   * @var \Symfony\Component\HttpFoundation\Request
   */
  protected $request;

  /**
   * Constructs a new PostmanCollectionController object.
   */
  public function __construct(APIManager $apiManager, RequestStack $requestStack) {
    $this->apiManager = $apiManager;
    $this->request = $requestStack->getCurrentRequest();
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('plugin.manager.rest_easy.api'),
      $container->get('request_stack')
    );
  }

  /**
   * List Postman collections for all REST Easy APIs.
   *
   * @return string[][]
   *   The render array for the collections page.
   */
  public function collections(): array {
    $render['description']['#markup'] = <<<DESCRIPTION

    <p><a href="https://www.postman.com/collection/" target="_blank">Postman collections</a> are a way to group and organize your API requests. You can import the collections into <a href="https://www.postman.com/downloads/" target="_blank">Postman</a> and use them to test your REST Easy API endpoints.</p>
    
    DESCRIPTION;

    $render['collections'] = [
      '#type' => 'table',
      '#header' => [
        'api' => $this->t('API'),
        'collection' => $this->t('Collection'),
      ],
      '#empty' => $this->t('No collections available.'),
      '#rows' => [],
    ];
    foreach ($this->apiManager->getDefinitions() as $pluginId => $apiPlugin) {
      $render['collections']['#rows'][] = [
        'api' => $apiPlugin['label'],
        'collection' => Link::fromTextAndUrl($this->t('Download'), Url::fromRoute('rest_easy_postman.download', ['plugin_id' => $pluginId])),
      ];
    }

    return $render;
  }

  /**
   * Download the Postman collection for a given API.
   *
   * @param string $plugin_id
   *   The ID of the API plugin.
   */
  public function download(string $plugin_id) {

    // Define the collection JSON.
    $api = $this->apiManager->createInstance($plugin_id);
    $collection = [
      'info' => [
        'name' => $api->getApiName(),
        'description' => $api->getDescription(),
        'schema' => 'https://schema.getpostman.com/json/collection/v2.1.0/collection.json',
      ],
      'variable' => [
        [
          'key' => 'baseUrl',
          'value' => $this->request->getSchemeAndHttpHost() . $api->getBasePath(),
        ],
      ],
    ];

    // Add the endpoints to the collection.
    $tags = $api->getTags();
    $tags[] = ['name' => $this->t('Other'), 'description' => $this->t('Endpoints that do not have a tag.')];
    foreach ($tags as $tag) {
      $endpoints = [];
      foreach ($api->endpoints as $endpoint) {
        $endpointDefinition = $endpoint->getPluginDefinition();
        if (in_array($tag['name'], $endpointDefinition['tags'])) {
          foreach ($endpointDefinition['methods'] as $method) {
            $endpoint = [
              'name' => $endpointDefinition['label'],
              'request' => [
                'method' => $method,
                'header' => [
                  [
                    'key' => 'Content-Type',
                    'value' => 'application/json',
                  ],
                ],
                'url' => [
                  'raw' => '{{baseUrl}}' . $endpointDefinition['path'],
                  'host' => [
                    '{{baseUrl}}',
                  ],
                  'path' => explode('/', substr($endpointDefinition['path'], 1)),
                ],
              ],
            ];
            foreach ($endpointDefinition['parameters'] as $parameter) {
              $parameterDefinition = $api->parameters[$parameter]->getPluginDefinition();
              if ($parameterDefinition['in'] == 'query') {
                $endpoint['request']['url']['query'][] = [
                  'key' => $parameterDefinition['id'],
                  'description' => $parameterDefinition['description'],
                  'disabled' => !$parameterDefinition['required'],
                  'value' => $parameterDefinition['default'] ?? '',
                ];
              }
            }
            $endpoints[] = $endpoint;
          }
        }
      }
      if ($endpoints) {
        $collection['item'][] = [
          'name' => $tag['name'],
          'description' => $tag['description'],
          'item' => $endpoints,
        ];
      }
    }

    // Download the collection with the appropriate Postman file extension.
    $response = new Response();
    $response->headers->set('Content-Disposition', 'attachment; filename=' . $plugin_id . '.postman_collection');
    $response->headers->set('Content-Type', 'application/json');
    $response->setContent(json_encode($collection, JSON_PRETTY_PRINT));
    return $response;
  }

}
