<?php

namespace Drupal\rest_easy\Plugin;

use Drupal\Component\Plugin\PluginBase;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\rest_easy\Event\APIRoutesEvent;
use Drupal\rest_easy\Event\APISecurityEvent;
use Drupal\rest_easy\Event\APISecurityDefinitionsEvent;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;

/**
 * A base class for creating REST Easy API plugins.
 *
 * @package Drupal\rest_easy\Plugin
 */
abstract class APIBase extends PluginBase implements ContainerFactoryPluginInterface, APIInterface {

  use StringTranslationTrait;

  /**
   * An array of endpoint plugins registered to the API.
   *
   * @var \Drupal\rest_easy\Plugin\EndpointInterface[]
   */
  public array $endpoints = [];

  /**
   * Event dispatcher service.
   *
   * @var \Symfony\Component\EventDispatcher\EventDispatcherInterface
   */
  public EventDispatcherInterface $eventDispatcher;

  /**
   * An array of parameter plugins registered to the API.
   *
   * @var \Drupal\rest_easy\Plugin\ParameterInterface[]
   */
  public array $parameters = [];

  /**
   * Schema definitions to include in OpenAPI documentation.
   *
   * @var array
   */
  protected array $schemaDefinitions = [
    'Status' => [
      'description' => 'A status message',
      'properties' => [
        'code' => [
          'description' => 'The numeric HTTP response status code. See https://www.w3.org/Protocols/rfc2616/rfc2616-sec10.html for details.',
          'type' => 'integer',
        ],
        'message' => [
          'description' => 'A human-readable message explaining the status',
          'type' => 'string',
        ],
        'errors' => [
          'description' => 'Errors that prevented the API call from executing, if applicable',
          'items' => ['type' => 'string'],
          'type' => 'array',
        ],
      ],
      'type' => 'object',
    ],
  ];

  /**
   * {@inheritdoc}
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, EventDispatcherInterface $eventDispatcher, LoggerChannelFactoryInterface $loggerFactory, EndpointManager $endpointManager, ParameterManager $parameterManager) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->eventDispatcher = $eventDispatcher;

    // Load endpoint plugins.
    $api_id = $this->getPluginId();
    foreach ($endpointManager->getDefinitions() as $endpoint_plugin_id => $endpoint_definition) {
      if ($endpoint_definition['api'] == $api_id) {
        try {
          $this->endpoints[$endpoint_plugin_id] = $endpointManager->createInstance($endpoint_plugin_id);
        }
        catch (\Exception $exception) {
          $loggerFactory->get('rest_easy')
            ->error($this->t('Unable to load the REST Easy endpoint plugin "@plugin_id": @message', [
              '@message' => $exception->getMessage(),
              '@plugin_id' => $endpoint_plugin_id,
            ]));
        }
      }
    }

    // Load parameter plugins.
    foreach ($parameterManager->getDefinitions() as $parameter_plugin_id => $parameter_definition) {
      if ($parameter_definition['api'] == $api_id) {
        try {
          $this->parameters[$parameter_plugin_id] = $parameterManager->createInstance($parameter_plugin_id);
        }
        catch (\Exception $exception) {
          $loggerFactory->get('rest_easy')
            ->error($this->t('Unable to load the REST Easy parameter plugin "@plugin_id": @message', [
              '@message' => $exception->getMessage(),
              '@plugin_id' => $parameter_plugin_id,
            ]));
        }
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('event_dispatcher'),
      $container->get('logger.factory'),
      $container->get('plugin.manager.rest_easy.endpoint'),
      $container->get('plugin.manager.rest_easy.parameter')
    );
  }

  /**
   * Retrieve the API name for OpenAPI documentation.
   *
   * @return string
   *   The API name
   *
   * @see \Drupal\rest_easy\Plugin\openapi\OpenApiGenerator\RESTEasyApiGenerator::getApiName()
   */
  public function getApiName(): string {
    $definition = $this->getPluginDefinition();
    return $definition['label'];
  }

  /**
   * Retrieve the API base path for OpenAPI documentation.
   *
   * @return string
   *   The API base path
   *
   * @see \Drupal\rest_easy\Plugin\openapi\OpenApiGenerator\RESTEasyApiGenerator::getBasePath()
   */
  public function getBasePath(): string {
    $definition = $this->getPluginDefinition();
    return $definition['base_path'];
  }

  /**
   * Retrieve the API description for OpenAPI documentation.
   *
   * @return string
   *   The API description.
   *
   * @see \Drupal\rest_easy\Plugin\openapi\OpenApiGenerator\RESTEasyApiGenerator::getApiDescription()
   */
  public function getDescription(): string {
    $definition = $this->getPluginDefinition();
    return $definition['description'];
  }

  /**
   * Generate API information for OpenAPI documentation.
   *
   * @return array
   *   An associative array consisting of a description, title, and version.
   *
   * @see \Drupal\rest_easy\Plugin\openapi\OpenApiGenerator\RESTEasyApiGenerator::getInfo()
   */
  public function getInfo(): array {
    $definition = $this->getPluginDefinition();
    return [
      'description' => $definition['description'],
      'title' => $definition['label'],
      'version' => $definition['version'],
    ];
  }

  /**
   * Retrieve the JSON schema for OpenAPI documentation.
   *
   * @param string $describedFormat
   *   The format that will be described, json, json_api, etc.
   * @param string $entityTypeId
   *   The entity type id.
   * @param string $bundleName
   *   The bundle name.
   *
   * @return array
   *   The JSON schema.
   *
   * @see \Drupal\rest_easy\Plugin\openapi\OpenApiGenerator\RESTEasyApiGenerator::getJsonSchema()
   */
  public function getJsonSchema($describedFormat, $entityTypeId, $bundleName = NULL) {
    return [];
  }

  /**
   * Assemble a list of shared parameters for OpenAPI documentation.
   *
   * @return array
   *   An associative array of shared parameters
   *
   * @see \Drupal\rest_easy\Plugin\openapi\OpenApiGenerator\RESTEasyApiGenerator::getSpecification()
   */
  public function getParameters(): array {
    $body_parameters = [];
    $parameters = [];
    foreach ($this->parameters as $parameter_plugin) {
      $parameter = [];
      $parameter_definition = $parameter_plugin->getPluginDefinition();
      foreach ($parameter_definition as $key => $value) {
        switch ($key) {
          case 'api':
          case 'class':
          case 'deriver':
          case 'provider':
            break;

          case 'default':
            if ($value !== NULL) {
              $parameter['default'] = $value;
            }
            break;

          case 'description':
            if ($value !== NULL) {
              $parameter['description'] = $value->render();
            }
            break;

          case 'enum':
            if (count($value) > 0) {
              $parameter['enum'] = $value;
            }
            break;

          case 'exclusiveMaximum':
          case 'exclusiveMinimum':
            if ($value) {
              $parameter[$key] = $value;
            }
            break;

          case 'id':
            $parameter['name'] = $value;
            break;

          case 'in':
            if ($value == 'body') {
              $body_parameters[] = $parameter_definition['id'];
            }
            elseif ($value !== NULL) {
              $parameter[$key] = $value;
            }
            break;

          case 'required':
            $parameter['required'] = $value;
            break;

          case 'type':
            if ($value !== NULL) {
              $parameter['type'] = $value;
            }
            break;

          default:
            if ($value !== NULL) {
              $parameter[$key] = $value;
            }
            break;
        }
      }
      $parameters[$parameter_definition['id']] = $parameter;
    }
    if ($body_parameters) {
      foreach ($this->endpoints as $endpoint) {
        $endpoint_definition = $endpoint->getPluginDefinition();
        if (isset($endpoint_definition['parameters'][0])) {
          foreach ($body_parameters as $body_parameter) {
            if (in_array($body_parameter, $endpoint_definition['parameters'])) {
              $endpoint_body_id = $endpoint_definition['id'] . '_body';
              if (!isset($parameters[$endpoint_body_id])) {
                $parameters[$endpoint_body_id] = [
                  'description' => $this->t('The request body.'),
                  'in' => 'body',
                  'name' => $endpoint_body_id,
                  'schema' => [
                    'properties' => [],
                    'type' => 'object',
                  ],
                ];
              }
              $parameters[$endpoint_body_id]['schema']['properties'][$body_parameter]['$ref'] = '#/parameters/' . $body_parameter;
            }
          }
        }
      }
    }
    return $parameters;
  }

  /**
   * Assemble paths for OpenAPI documentation.
   *
   * @return array
   *   An associative array of paths and methods.
   *
   * @see \Drupal\rest_easy\Plugin\openapi\OpenApiGenerator\RESTEasyApiGenerator::getPaths()
   */
  public function getPaths(): array {
    $paths = [];
    foreach ($this->endpoints as $endpoint) {
      $endpoint_definition = $endpoint->getPluginDefinition();
      $path = $endpoint->path();
      if (isset($path['parameters'][0])) {
        foreach ($path['parameters'] as $key => $parameter) {
          $endpoint_body_id = '#/parameters/' . $endpoint_definition['id'] . '_body';
          if ($parameter['$ref'] == $endpoint_body_id) {
            continue;
          }
          $parameter = $this->parameters[substr($parameter['$ref'], 13)]->getPluginDefinition();
          if (isset($parameter['in']) && $parameter['in'] == 'body') {
            unset($path['parameters'][$key]);
            $path['parameters'][] = ['$ref' => $endpoint_body_id];
          }
        }
        $path['parameters'] = array_values($path['parameters']);
      }
      foreach ($endpoint_definition['methods'] as $method) {
        $paths[$endpoint_definition['path']][strtolower($method)] = $path;
      }
    }
    return $paths;
  }

  /**
   * Assemble schema definitions for OpenAPI documentation.
   *
   * @return array[]
   *   An array of schema definitions.
   *
   * @see \Drupal\rest_easy\Plugin\openapi\OpenApiGenerator\RESTEasyApiGenerator::getSchemaDefinitions()
   */
  public function getSchemaDefinitions(): array {
    $definitions = $this->schemaDefinitions;
    foreach ($this->endpoints as $endpoint) {
      $definitions = array_merge($definitions, $endpoint->schemaDefinitions());
    }
    ksort($definitions);
    return $definitions;
  }

  /**
   * Assemble security definitions for OpenAPI documentation.
   *
   * @return array
   *   An associative array of security definitions.
   *
   * @see \Drupal\rest_easy\Plugin\openapi\OpenApiGenerator\RESTEasyApiGenerator::getSecurityDefinitions()
   */
  public function getSecurityDefinitions(): array {
    $definitions = [];
    $event = new APISecurityDefinitionsEvent($this, $definitions);
    $this->eventDispatcher->dispatch($event, APISecurityDefinitionsEvent::EVENT_NAME);
    return $event->definitions;
  }

  /**
   * Assemble security entries for OpenAPI documentation.
   *
   * @return array
   *   An associative array of security entries.
   *
   * @see \Drupal\rest_easy\Plugin\openapi\OpenApiGenerator\RESTEasyApiGenerator::getSecurity()
   */
  public function getSecurity() {
    $security = [];

    // Dispatch the event to allow other modules to alter security entries.
    $event = new APISecurityEvent($this, $security);
    $this->eventDispatcher->dispatch($event, APISecurityEvent::EVENT_NAME);
    return $event->security;
  }

  /**
   * Retrieve API tags for OpenAPI documentation.
   *
   * @return array
   *   An array of tag names and descriptions
   *
   * @see \Drupal\rest_easy\Plugin\openapi\OpenApiGenerator\RESTEasyApiGenerator::getTags()
   */
  public function getTags(): array {
    $definition = $this->getPluginDefinition();
    $tags = [];
    if (isset($definition['tags']) && is_array($definition['tags']) && count($definition['tags']) > 0) {
      foreach ($definition['tags'] as $tag => $description) {
        $tags[] = ['name' => $tag, 'description' => $description];
      }
    }
    return $tags;
  }

  /**
   * Generate routes for the API.
   *
   * @return array
   *   An array of routes.
   *
   * @see \Drupal\rest_easy\Routing\APIRouteBuilder::routes()
   */
  public function routes(): array {
    $definition = $this->getPluginDefinition();
    $routes = [];
    foreach ($this->endpoints as $endpoint_plugin) {
      $route = $endpoint_plugin->route();

      // Prepend the API's base path.
      $route->setPath($definition['base_path'] . $route->getPath());

      // Use the API's authentication plugin list if the endpoint doesn't have
      // its own.
      if ($definition['auth']) {
        $options = $route->getOptions();
        if (!isset($options['_auth'])) {
          $options['_auth'] = $definition['auth'];
          $route->setOptions($options);
        }
      }

      $routes['rest_easy.' . $definition['id'] . '.' . $endpoint_plugin->getPluginId()] = $route;
    }

    // Dispatch the event to allow other modules to alter the routes.
    $event = new APIRoutesEvent($this, $routes);
    $this->eventDispatcher->dispatch($event, APIRoutesEvent::EVENT_NAME);
    return $event->routes;
  }

}
