<?php

namespace Drupal\rest_easy\Plugin;

use Drupal\Component\Plugin\PluginInspectionInterface;

/**
 * An interface for creating REST Easy API plugins.
 *
 * @package Drupal\rest_easy\Plugin
 */
interface APIInterface extends PluginInspectionInterface {

  /**
   * Retrieve the API name for OpenAPI documentation.
   *
   * @return string
   *   The API name
   */
  public function getApiName(): string;

  /**
   * Retrieve the API base path for OpenAPI documentation.
   *
   * @return string
   *   The API base path
   */
  public function getBasePath(): string;

  /**
   * Retrieve the API description for OpenAPI documentation.
   *
   * @return string
   *   The API description.
   */
  public function getDescription(): string;

  /**
   * Generate API information for OpenAPI documentation.
   *
   * @return array
   *   The API information
   */
  public function getInfo(): array;

  /**
   * Retrieve the JSON schema for OpenAPI documentation.
   *
   * @param string $describedFormat
   *   The format that will be described, json, json_api, etc.
   * @param string $entityTypeId
   *   The entity type id.
   * @param string $bundleName
   *   The bundle name.
   *
   * @return array
   *   The JSON schema.
   *
   * @see \Drupal\rest_easy\Plugin\openapi\OpenApiGenerator\RESTEasyApiGenerator::getJsonSchema()
   */
  public function getJsonSchema($describedFormat, $entityTypeId, $bundleName = NULL);

  /**
   * Assemble a list of shared parameters for OpenAPI documentation.
   *
   * @return array
   *   An associative array of shared parameters
   *
   * @see \Drupal\rest_easy\Plugin\openapi\OpenApiGenerator\RESTEasyApiGenerator::getSpecification()
   */
  public function getParameters(): array;

  /**
   * Assemble paths for OpenAPI documentation.
   *
   * @return array
   *   An associative array of paths and methods.
   */
  public function getPaths(): array;

  /**
   * Assemble schema definitions for OpenAPI documentation.
   *
   * @return array[]
   *   An array of schema definitions.
   *
   * @see \Drupal\rest_easy\Plugin\openapi\OpenApiGenerator\RESTEasyApiGenerator::getSchemaDefinitions()
   */
  public function getSchemaDefinitions(): array;

  /**
   * Assemble security definitions for OpenAPI documentation.
   *
   * @return array
   *   An associative array of security definitions.
   *
   * @see \Drupal\rest_easy\Plugin\openapi\OpenApiGenerator\RESTEasyApiGenerator::getSecurityDefinitions()
   */
  public function getSecurityDefinitions(): array;

  /**
   * Generate API tags for OpenAPI documentation.
   *
   * @return array
   *   An array of tag names and descriptions
   */
  public function getTags(): array;

  /**
   * Generate routes for the API.
   *
   * @return array
   *   An array of routes.
   *
   * @see \Drupal\rest_easy\Routing\APIRouteBuilder::routes()
   */
  public function routes(): array;

}
