<?php

namespace Drupal\rest_easy\Plugin;

use Drupal\Component\Plugin\PluginBase;
use Drupal\Core\Access\AccessResult;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\rest_easy\Controller\EndpointController;
use Drupal\rest_easy\Event\EndpointAccessEvent;
use Drupal\rest_easy\Event\EndpointCallEvent;
use Drupal\rest_easy\Event\EndpointHeadersEvent;
use Drupal\rest_easy\Event\EndpointRouteEvent;
use Drupal\rest_easy\Traits\HTTPCodesTrait;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\RequestStack;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Route;

/**
 * A base class for building REST Easy endpoint plugins.
 *
 * @package Drupal\rest_easy\Plugin
 */
abstract class EndpointBase extends PluginBase implements ContainerFactoryPluginInterface, EndpointInterface {

  use HTTPCodesTrait;
  use StringTranslationTrait;

  /**
   * A list of domains allowed to access the API.
   *
   * @var array
   */
  protected mixed $allowedOrigins = [];

  /**
   * Current user.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface
   */
  protected AccountProxyInterface $currentUser;

  /**
   * Event dispatcher service.
   *
   * @var \Symfony\Component\EventDispatcher\EventDispatcherInterface
   */
  public EventDispatcherInterface $eventDispatcher;

  /**
   * Logger service.
   *
   * @var \Drupal\Core\Logger\LoggerChannelInterface
   */
  protected LoggerChannelInterface $logger;

  /**
   * REST Easy parameter manager service.
   *
   * @var ParameterManager
   */
  protected ParameterManager $parameterManager;

  /**
   * An associative array of parameters passed to the endpoint.
   *
   * @var array
   */
  protected array $parameters = [];

  /**
   * Current request.
   *
   * @var \Symfony\Component\HttpFoundation\Request
   */
  protected Request $request;

  /**
   * Constructs a new EndpointBase object.
   *
   * @param array $configuration
   *   Plugin configuration.
   * @param mixed $plugin_id
   *   Plugin ID.
   * @param mixed $plugin_definition
   *   Plugin definition.
   * @param \Drupal\rest_easy\Plugin\APIManager $apiManager
   *   REST Easy API plugin manager service.
   * @param \Drupal\Core\Session\AccountProxyInterface $currentUser
   *   Current user.
   * @param \Symfony\Component\EventDispatcher\EventDispatcherInterface $eventDispatcher
   *   Event dispatcher service.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $loggerFactory
   *   Logger channel factory service.
   * @param \Drupal\rest_easy\Plugin\ParameterManager $parameterManager
   *   REST Easy parameter manager service.
   * @param \Symfony\Component\HttpFoundation\RequestStack $requestStack
   *   Request stack.
   * @param \Drupal\Core\Routing\RouteMatchInterface $routeMatch
   *   Current route match.
   *
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, APIManager $apiManager, AccountProxyInterface $currentUser, EventDispatcherInterface $eventDispatcher, LoggerChannelFactoryInterface $loggerFactory, ParameterManager $parameterManager, RequestStack $requestStack, RouteMatchInterface $routeMatch) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->currentUser = $currentUser;
    $this->eventDispatcher = $eventDispatcher;
    $this->logger = $loggerFactory->get('rest_easy');
    $this->parameterManager = $parameterManager;
    $this->request = $requestStack->getCurrentRequest();

    // Define allowed origins.
    $endpoint_definition = $this->getPluginDefinition();
    if ($endpoint_definition['allowed_origins']) {
      $this->allowedOrigins = $endpoint_definition['allowed_origins'];
    }
    else {
      $api_definition = $apiManager->getDefinition($endpoint_definition['api']);
      if (isset($api_definition['allowed_origins']) && $api_definition['allowed_origins']) {
        $this->allowedOrigins = $api_definition['allowed_origins'];
      }
    }

    // Always allow the origin on which the API is running.
    if ($this->allowedOrigins) {
      $current_origin = $this->request->getSchemeAndHttpHost();
      if (!in_array($current_origin, $this->allowedOrigins)) {
        array_unshift($this->allowedOrigins, $current_origin);
      }
    }

    // Parse and validate parameters.
    if (count($endpoint_definition['parameters']) > 0 && $routeMatch->getRouteName() == 'rest_easy.' . $endpoint_definition['api'] . '.' . $plugin_id) {
      $errors = [];
      foreach ($endpoint_definition['parameters'] as $parameter_id) {
        try {
          $this->parameters[$parameter_id] = NULL;
          $parameter = $this->parameterManager->createInstance($parameter_id);
          $parameter_value = $parameter->get();
          $parameter_errors = $parameter->validate($parameter_value);
          if (count($parameter_errors) > 0) {
            $errors = array_merge($errors, $parameter_errors);
          }
          else {
            $this->parameters[$parameter_id] = $parameter_value;
          }
        }
        catch (\Exception $exception) {
          $errors[] = $exception->getMessage();
        }
      }
      if ($errors) {
        $this::status(400, 'Bad Request', ['errors' => $errors]);
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('plugin.manager.rest_easy.api'),
      $container->get('current_user'),
      $container->get('event_dispatcher'),
      $container->get('logger.factory'),
      $container->get('plugin.manager.rest_easy.parameter'),
      $container->get('request_stack'),
      $container->get('current_route_match')
    );
  }

  /**
   * Determine if a user can access the endpoint.
   *
   * By default, this is true if a user has the permission specified in the
   * endpoint plugin annotation. If no permission is supplied, it will back up
   * to the permission supplied by the API plugin annotation. If neither
   * specify a permission, any request will be allowed. Override this method
   * for endpoints with more sophisticated access handling.
   *
   * @param array $arguments
   *   An associative array of path arguments.
   *
   * @return \Drupal\Core\Access\AccessResult
   *   Whether the user can access the endpoint.
   */
  public function access(...$arguments): AccessResult {

    // Use the permission defined by the endpoint plugin, if any.
    $result = NULL;
    $endpoint_definition = $this->getPluginDefinition();
    if (isset($endpoint_definition['permission']) && $endpoint_definition['permission']) {
      $result = AccessResult::allowedIfHasPermission($this->currentUser, $endpoint_definition['permission']);
    }

    // Default to the permission defined by the API plugin, if any.
    else {
      $api_definition = \Drupal::service('plugin.manager.rest_easy.api')->getDefinition($endpoint_definition['api']);
      if (isset($api_definition['permission']) && $api_definition['permission']) {
        $result = AccessResult::allowedIfHasPermission($this->currentUser, $api_definition['permission']);
      }
    }

    // If no permission is defined, allow all access.
    if ($result === NULL) {
      $result = AccessResult::allowed();
    }

    // Dispatch an event to allow other modules to alter the access result.
    $event = new EndpointAccessEvent($this, $result);
    $this->eventDispatcher->dispatch($event, EndpointAccessEvent::EVENT_NAME);
    return $event->result;
  }

  /**
   * Execute the endpoint and return the appropriate response.
   *
   * @return mixed
   *   The endpoint's response.
   */
  abstract public function call(...$arguments): mixed;

  /**
   * Generate HTTP headers for the endpoint's responses.
   *
   * @return array
   *   An associative array of HTTP headers
   */
  public function headers(): array {

    // Define the Access-Control-Allow-Methods header.
    $definition = $this->getPluginDefinition();
    $headers = [];
    if (isset($definition['methods'][0])) {
      $headers['Access-Control-Allow-Methods'] = implode(', ', $definition['methods']);
    }

    // Define the Access-Control-Allow-Origin header.
    $headers['Access-Control-Allow-Origin'] = $definition['origins'] ?? '*';
    if (!$this->allowedOrigins) {
      $headers['Access-Control-Allow-Origin'] = '*';
    }
    else {
      $origin = FALSE;
      foreach ($this->allowedOrigins as $allowed_origin) {
        if ($this->request->headers->get('Origin') == $allowed_origin) {
          $origin = $allowed_origin;
          break;
        }
      }
      $headers['Access-Control-Allow-Origin'] = $origin ?: $this->allowedOrigins[0];
    }

    // Define the Cache-Control and Expires headers.
    if ($definition['cache_lifetime']) {
      $expires = time() + $definition['cache_lifetime'];
      $headers['Cache-Control'] = 'max-age=' . $expires;
      $headers['Expires'] = gmdate('D, d M Y H:i:s T', $expires);
    }
    else {
      $headers['Cache-Control'] = 'no-store';
      $headers['Expires'] = gmdate('D, d M Y H:i:s T', 0);
    }

    // Dispatch an event to allow other modules to alter the headers.
    $event = new EndpointHeadersEvent($this, $headers);
    $this->eventDispatcher->dispatch($event, EndpointHeadersEvent::EVENT_NAME);
    return $event->headers;
  }

  /**
   * Package the data into a JSON response.
   *
   * @param mixed $data
   *   The data to include in the response.
   *
   * @return \Symfony\Component\HttpFoundation\Response
   *   The JSON response.
   */
  public function jsonResponse($data): Response {

    // Construct the JSON response object.
    $headers = $this->headers();
    $headers['Content-Type'] = 'application/json';
    $response = new Response(json_encode($data), 200, $headers);

    // Dispatch an event to allow other modules to alter the response.
    $event = new EndpointCallEvent($this, $response);
    $this->eventDispatcher->dispatch($event, EndpointCallEvent::EVENT_NAME);
    return $event->response;
  }

  /**
   * Generate path data for OpenAPI documentation.
   *
   * @return array
   *   An array containing structured path information.
   */
  public function path(): array {
    $definition = $this->getPluginDefinition();
    $path = [];
    if ($definition['label']) {
      $path['summary'] = $definition['label'];
    }
    if ($definition['description']) {
      $path['description'] = $definition['description'];
    }
    if ($definition['parameters']) {
      foreach ($definition['parameters'] as $parameter) {
        $path['parameters'][] = ['$ref' => '#/parameters/' . $parameter];
      }
    }
    if ($definition['responses']) {
      foreach ($definition['responses'] as $http_status_code => $schema_reference) {
        if (is_int($http_status_code) && $http_status_code < 100) {
          $http_status_code = $schema_reference;
          $schema_reference = '#/definitions/Status';
        }
        else {
          $schema_reference = '#/definitions/' . $schema_reference;
        }
        $response = [];
        if (isset(self::HTTP_CODES[$http_status_code])) {
          $response['description'] = self::HTTP_CODES[$http_status_code];
        }
        $response['schema']['$ref'] = $schema_reference;
        $path['responses'][$http_status_code] = $response;
      }
    }
    if (isset($definition['auth'][0]) && in_array('basic_auth', $definition['auth'])) {
      $path['security'] = ['basicAuth' => []];
    }
    if ($definition['tags']) {
      $path['tags'] = $definition['tags'];
    }
    return $path;
  }

  /**
   * Generate the route for the endpoint.
   *
   * @return \Symfony\Component\Routing\Route
   *   The route object.
   *
   * @see \Drupal\rest_easy\Plugin\APIBase::routes()
   * @see \Drupal\rest_easy\Routing\APIRouteBuilder::routes()
   */
  public function route(): Route {
    $definition = $this->getPluginDefinition();

    // Define route defaults.
    $defaults = [
      '_controller' => EndpointController::class . '::call',
      'plugin_id' => $definition['id'],
    ];

    // Define route options.
    $options = [];
    if ($definition['auth']) {
      $options['_auth'] = $definition['auth'];
    }

    // @todo Add parameter entity typing to the options array.
    // Define route requirements.
    $requirements = [
      '_custom_access' => EndpointController::class . '::access',
    ];
    foreach ($definition['parameters'] as $parameter) {
      $parameter_definition = $this->parameterManager->getDefinition($parameter);
      if ($parameter_definition['in'] == 'path' && $parameter_definition['pattern']) {
        $requirements[$definition['id']] = $parameter_definition['pattern'];
      }
    }

    // Construct the route object.
    $route = new Route($definition['path'], $defaults, $requirements, $options, '', [], $definition['methods']);

    // Dispatch an event to allow other modules to alter the route.
    $event = new EndpointRouteEvent($this, $route);
    $this->eventDispatcher->dispatch($event, EndpointRouteEvent::EVENT_NAME);
    return $event->route;
  }

  /**
   * Generate a list of schema definitions for the OpenAPI documentation.
   *
   * @return array
   *   An associative array of schema definitions
   *
   * @see https://swagger.io/specification/v2/#definitions-object-example
   */
  public function schemaDefinitions(): array {
    return [];
  }

  /**
   * Generate an HTTP response along with a status message.
   *
   * @param int $code
   *   The numeric HTTP response status code to return. Common codes include:
   *    - 200 to indicate success
   *    - 201 to indicate that a record was successfully created
   *    - 400 to indicate a bad request, such as an invalid parameter value
   *    - 401 to indicate missing authentication
   *    - 403 to indicate insufficient access rights
   *    - 404 to indicate a missing record
   *    - 405 to indicate an unsupported method (e.g., POST, DELETE)
   * @param string $message
   *   The message to include in the response.
   * @param array $additionalData
   *   An associative array of additional data to pass with the response.
   *
   * @see HTTPResponseTrait
   */
  public static function status(int $code, string $message = '', array $additionalData = []): void {
    if (!$message) {
      $message = self::HTTP_CODES[$code] ?? '';
    }
    $response = new Response(json_encode(array_merge([
      'code' => $code,
      'message' => $message,
    ], $additionalData)), $code, [
      'Access-Control-Allow-Origin' => '*',
      'Content-Type' => 'application/json',
    ]);
    $response->send();
    exit();
  }

}
