<?php

namespace Drupal\rest_easy\Traits;

use Drupal\rest_easy\Event\EndpointCallEvent;
use Symfony\Component\HttpFoundation\Response;

/**
 * A trait to add to REST Easy endpoint plugins to return CSV responses.
 *
 * @package Drupal\rest_easy\Trait
 */
trait CSVEndpointTrait {

  /**
   * Package the data into a CSV response.
   *
   * @param array $data
   *   The data to include in the response.
   *
   * @return \Symfony\Component\HttpFoundation\Response
   *   The CSV response.
   */
  public function csvResponse(array $data): Response {

    // Construct the CSV response object.
    $http_headers = $this->headers();
    $http_headers['Content-Type'] = 'application/csv';
    if (!str_starts_with($this->request->server->get('HTTP_REFERER'), '/admin/config/services/openapi/swagger')) {
      $http_headers['Content-Disposition'] = 'attachment; filename=export.csv';
    }
    $response = new Response('', 200, $http_headers);

    // Parse the contents into a CSV.
    $csv_headers = $rows = [];
    foreach ($data as $row) {
      $row = $this::flatten($row);
      $rows[] = $row;
      foreach ($row as $column => $value) {
        if (!in_array($column, $csv_headers)) {
          $csv_headers[] = $column;
        }
      }
    }
    $csv = fopen('php://temp', 'w+');
    fputcsv($csv, $csv_headers);
    foreach ($rows as $row) {
      $values = [];
      foreach ($csv_headers as $header) {
        if (isset($row[$header])) {
          $values[] = $row[$header];
        }
        else {
          $values[] = '';
        }
      }
      fputcsv($csv, $values);
    }
    rewind($csv);
    $response->setContent(stream_get_contents($csv));
    fclose($csv);

    // Dispatch an event to allow other modules to alter the response.
    $event = new EndpointCallEvent($this, $response);
    $this->eventDispatcher->dispatch($event, EndpointCallEvent::EVENT_NAME);
    return $event->response;
  }

  /**
   * Recursively flatten an array while preserving prefixed keys.
   *
   * @param array $array
   *   The array to be flattened.
   * @param string $prefix
   *   The prefix to apply to the array key.
   *
   * @return array
   *   The flattened array.
   *
   * @see https://stackoverflow.com/a/9546215
   */
  protected static function flatten($array, string $prefix = ''): array {
    $result = [];
    foreach ($array as $key => $value) {
      if (is_array($value)) {
        $result = $result + self::_flatten($value, $prefix . $key . '.');
      }
      else {
        $result[$prefix . $key] = $value;
      }
    }
    return $result;
  }

}
