<?php

namespace Drupal\rest_easy\Routing;

use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\rest_easy\Plugin\APIManager;

/**
 * An object for dynamically generating REST Easy endpoint routes.
 *
 * @package Drupal\rest_easy\Routing
 */
class APIRouteBuilder {

  use StringTranslationTrait;

  /**
   * REST Easy API plugin manager service.
   *
   * @var \Drupal\rest_easy\Plugin\APIManager
   */
  protected APIManager $apiManager;

  /**
   * Logger service.
   *
   * @var \Drupal\Core\Logger\LoggerChannelInterface
   */
  protected LoggerChannelInterface $logger;

  /**
   * Constructs a new APIRouteBuilder object.
   *
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $loggerFactory
   *   Logger factory service.
   * @param \Drupal\rest_easy\Plugin\APIManager $apiManager
   *   API manager service.
   */
  public function __construct(LoggerChannelFactoryInterface $loggerFactory, APIManager $apiManager) {
    $this->apiManager = $apiManager;
    $this->logger = $loggerFactory->get('rest_easy');
  }

  /**
   * Generate routes for every endpoint plugin.
   *
   * @return array
   *   The generated routes.
   */
  public function routes(): array {
    $routes = [];
    foreach ($this->apiManager->getDefinitions() as $plugin_id => $plugin_definition) {
      try {
        $api_plugin = $this->apiManager->createInstance($plugin_id);
        $routes = array_merge($routes, $api_plugin->routes());
      }
      catch (\Exception $exception) {
        $this->logger->error($this->t('Unable to generate routes for REST Easy API plugin "@plugin_id: @message', [
          '@message' => $exception->getMessage(),
          '@plugin_id' => $plugin_id,
        ]));
      }
    }
    return $routes;
  }

}
