<?php

namespace Drupal\rest_easy\Traits;

/**
 * A trait to add to REST Easy endpoint plugins to enable caching.
 *
 * @package Drupal\rest_easy\Trait
 */
trait CacheableEndpointTrait {

  /**
   * The plugin ID for a boolean parameter used to enable or disable caching.
   *
   * @var string
   */
  protected $cacheParameter = 'cache';

  /**
   * Generate a cache ID.
   *
   * @param int $index
   *   The backtrace index to use when generating the cache ID. This is set to
   *   2 by default, indicating the function prior to the calling function.
   *
   * @return string
   *   The generated cache ID.
   */
  protected function cid(int $index = 2): string {
    $backtrace = debug_backtrace(0, $index + 1);
    $cid_base = $backtrace[$index]['class'];
    $cid_base = substr($cid_base, strrpos($cid_base, '\\') + 1);
    $cid_base .= '::' . $backtrace[$index]['function'];
    $cid = '(';
    if (isset($backtrace[$index]['args'][0])) {
      $arguments = [];
      foreach ($backtrace[$index]['args'] as $argument) {
        if (is_scalar($argument)) {
          $arguments[] = $argument;
        }
        else {
          $arguments[] = md5(serialize($argument));
        }
      }
      $cid .= implode(',', $arguments) . ')';
    }
    $parameters = $this->parameters;
    $cid .= '?' . http_build_query($parameters);
    if (strlen($cid_base) + strlen($cid) > 255) {
      $cid = '(' . md5($cid) . ')';
    }
    return $cid_base . $cid;
  }

  /**
   * Get a cached response. Return false if there is no cached response.
   *
   * @return mixed|false
   *   The cached response or false if there is no cached response.
   */
  protected function cacheGet(): mixed {
    if ((!$this->cacheParameter || $this->parameters[$this->cacheParameter]) && ($cache = \Drupal::cache()->get($this->cid()))) {
      return $cache->data;
    }
    return FALSE;
  }

  /**
   * Cache a response.
   *
   * @param mixed $value
   *   The response to cache.
   * @param array $cache_tags
   *   Cache tags to invalidate the cache entry.
   */
  protected function cacheSet($value, array $cache_tags = []): void {
    $definition = $this->getPluginDefinition();
    $cache_lifetime = $definition['cache_lifetime'] ?? 3600;
    if ($cache_lifetime) {
      \Drupal::cache()->set($this->cid(), $value, time() + $cache_lifetime, $cache_tags);
    }
  }

  /**
   * Set the cache parameter used by the endpoint.
   *
   * @param string $parameter
   *   The plugin ID for a boolean parameter used to enable or disable caching.
   */
  protected function setCacheParameter(string $parameter): void {
    $this->cacheParameter = $parameter;
  }

}
