<?php

namespace Drupal\restaurant_order\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Link;
use Drupal\Core\Url;
use Symfony\Component\HttpFoundation\RedirectResponse;

/**
 * Menu items admin listing controller.
 */
class MenuItemController extends ControllerBase {

  /**
   *
   */
  public function list() {
    $build = [];

    // Fetch all menu items grouped by category.
    $query = \Drupal::database()->select('menu_item', 'm')
      ->fields('m')
      ->orderBy('category')
      ->orderBy('name');
    $result = $query->execute();

    $grouped_items = [];

    foreach ($result as $r) {
      $grouped_items[$r->category][] = $r;
    }

    // Add "Add Menu Item" button.
    $build['add'] = [
      '#type' => 'link',
      '#title' => $this->t('Add Menu Item'),
      '#url' => Url::fromRoute('restaurant_order.menu_item_add'),
      '#attributes' => ['class' => ['button', 'button--primary']],
    ];

    // Table headers.
    $header = [
      $this->t('ID'),
      $this->t('Name'),
      $this->t('Price'),
      $this->t('Type'),
      $this->t('Operations'),
    ];

    // Create a separate table per category.
    foreach ($grouped_items as $category => $items) {
      $rows = [];

      foreach ($items as $item) {
        $ops = [];
        $ops[] = Link::fromTextAndUrl(
          $this->t('Edit'),
          Url::fromRoute('restaurant_order.menu_item_edit', ['id' => $item->id])
        );
        $ops[] = Link::fromTextAndUrl(
          $this->t('Delete'),
          Url::fromRoute('restaurant_order.menu_item_delete', ['id' => $item->id])
        );

        $rows[] = [
          'data' => [
            $item->id,
            $item->name,
            '₹' . number_format($item->price, 2),
            $item->type,
            [
              'data' => [
                '#theme' => 'item_list',
                '#items' => $ops,
                '#type' => 'inline',
              ],
            ],
          ],
        ];
      }

      // Add heading and table for each category.
      $build['category_' . preg_replace('/[^a-z0-9_]+/', '_', strtolower($category))] = [
        '#type' => 'details',
        '#title' => $this->t('@category', ['@category' => $category]),
        '#open' => TRUE,
        'table' => [
          '#type' => 'table',
          '#header' => $header,
          '#rows' => $rows,
          '#attributes' => ['class' => ['menu-items-table']],
        ],
        '#attached' => [
          'library' => ['restaurant_order/styles'],
        ],
      ];
    }
    $build['#cache'] = ['max-age' => 0];

    return $build;
  }

  /**
   *
   */
  public function delete($id) {
    $db = \Drupal::database();

    $menu_item = $db->select('menu_item', 'm')
      ->fields('m')
      ->condition('id', $id)
      ->execute()
      ->fetchObject();

    if (!$menu_item) {
      $this->messenger()->addError($this->t('Menu item not found.'));
      return new RedirectResponse(Url::fromRoute('restaurant_order.menu_items')->toString());
    }

    // Delete the menu item.
    $db->delete('menu_item')->condition('id', $id)->execute();

    $this->messenger()->addStatus($this->t('Menu item "%name" has been deleted.', ['%name' => $menu_item->name]));

    // Redirect back to the menu items list.
    return new RedirectResponse(Url::fromRoute('restaurant_order.menu_items')->toString());
  }

}
