<?php

namespace Drupal\restaurant_order\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Link;
use Drupal\Core\Url;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;
use Symfony\Component\HttpFoundation\Response;
use Dompdf\Dompdf;

/**
 * Order controller: place order, list orders, print receipts, update status.
 */
class OrderController extends ControllerBase {

  /**
   * Place an order (order form).
   *
   * Optional $table_no may be passed (route or link). If provided we calculate
   * the appropriate session_id and pass both as default values to the form.
   */
  public function placeOrder($table_no = NULL) {
    $default_values = [];

    if ($table_no !== NULL) {
      // Determine active session id for this table (or new one).
      $session_id = $this->getActiveSessionForTable($table_no);
      $default_values = [
        'table_no' => $table_no,
        'session_id' => $session_id,
      ];
    }

    $form = \Drupal::formBuilder()->getForm('Drupal\\restaurant_order\\Form\\OrderForm', $default_values);
    $form['#cache'] = ['max-age' => 0];
    return $form;
  }

  /**
   * Autocomplete callback for menu items.
   */
  public function autocompleteItems(Request $request) {
    $results = [];
    $search = $request->query->get('q');

    if ($search !== NULL && $search !== '') {
      $query = \Drupal::database()->select('menu_item', 'm');
      $query->fields('m', ['id', 'name', 'price']);
      $query->condition('m.name', '%' . $search . '%', 'LIKE');
      $query->range(0, 10);

      $items = $query->execute()->fetchAll();

      foreach ($items as $item) {
        $results[] = [
          'value' => $item->name,
          'label' => $item->name . ' (₹' . number_format($item->price, 2) . ')',
        ];
      }
    }

    return new JsonResponse($results);
  }

  /**
   *
   */
  public function listOrders() {
    $build = [];

    $query = \Drupal::database()->select('restaurant_order', 'o');
    $query->fields('o', ['id', 'table_no', 'items', 'status', 'created', 'session_id']);
    $res = $query->execute();

    $orders = [];
    $max_id_per_table = [];
    $count_per_table = [];

    foreach ($res as $r) {
      $orders[] = $r;

      // Track max order ID per table-session.
      $key = $r->table_no . '-' . $r->session_id;
      if (!isset($max_id_per_table[$key]) || $r->id > $max_id_per_table[$key]) {
        $max_id_per_table[$key] = $r->id;
      }

      // Count orders per table-session.
      if (!isset($count_per_table[$key])) {
        $count_per_table[$key] = 0;
      }
      $count_per_table[$key]++;
    }

    $rows = [];
    foreach ($orders as $r) {
      $items = json_decode($r->items, TRUE) ?: [];
      $summary = [];
      foreach ($items as $it) {
        $summary[] = $it['name'] . ' x' . $it['qty'];
      }

      $operations = [];
      $operations[] = Link::fromTextAndUrl($this->t('Print KOT'),
      Url::fromRoute('restaurant_order.print_receipt', ['order_id' => $r->id, 'type' => 'kot']));
      $operations[] = Link::fromTextAndUrl($this->t('Print BOT'),
        Url::fromRoute('restaurant_order.print_receipt', ['order_id' => $r->id, 'type' => 'bot']));

      $key = $r->table_no . '-' . $r->session_id;
      $table_order_count = $count_per_table[$key];
      $is_last_order = ($r->id === $max_id_per_table[$key]);

      // Insert Bill link (saves to restaurant_bill).
      if ($is_last_order) {
        $operations[] = Link::fromTextAndUrl($this->t('Save Bill'),
          Url::fromRoute('restaurant_order.insert_bill', [
            'table_no' => $r->table_no,
            'session_id' => $r->session_id,
          ])
          );

        // Print aggregated bill only after inserting.
        $operations[] = Link::fromTextAndUrl($this->t('Print Aggregated Bill'),
            Url::fromRoute('restaurant_order.print_bill_by_table', [
              'table_no' => $r->table_no,
              'session_id' => $r->session_id,
            ])
          );
      }
      else {
        $operations[] = Link::fromTextAndUrl($this->t('Print Bill'),
          Url::fromRoute('restaurant_order.print_receipt', ['order_id' => $r->id, 'type' => 'bill']));
      }

      $operations[] = Link::fromTextAndUrl($this->t('View'),
        Url::fromRoute('restaurant_order.view_order', ['order_id' => $r->id]));

      $drop_links = array_reduce($operations, function ($carry, $link) {
        if ($link instanceof Link) {
          $title = $link->getText();
          $carry[strtolower(str_replace(' ', '_', $title))] = [
            'title' => $title,
            'url' => $link->getUrl(),
          ];
        }
        return $carry;
      }, []);

      $row_class = 'order-status-' . $r->status;

      $rows[] = [
        'data' => [
          $r->id,
          $r->table_no . ' (S' . $r->session_id . ')',
          implode(', ', $summary),
          ucfirst($r->status),
          \Drupal::service('date.formatter')->format($r->created, 'short'),
        [
          'data' => [
            '#type' => 'dropbutton',
            '#links' => $drop_links,
            '#attributes' => ['class' => ['order-actions-dropbutton']],
          ],
        ],
        ],
        'class' => [$row_class],
      ];
    }

    $build['table'] = [
      '#type' => 'table',
      '#header' => [
        $this->t('#'),
        $this->t('Table'),
        $this->t('Items'),
        $this->t('Status'),
        $this->t('Created'),
        $this->t('Actions'),
      ],
      '#rows' => $rows,
      '#attributes' => [
        'class' => ['table', 'table-striped', 'table-hover', 'table-sm'],
      ],
      '#attached' => [
        'library' => ['restaurant_order/styles'],
      ],
    ];

    $build['#cache'] = ['max-age' => 0];
    return $build;
  }

  /**
   * Print receipt: KOT, BOT, Bill.
   */
  public function printReceipt($order_id, $type) {
    $db = \Drupal::database();
    $order = $db->select('restaurant_order', 'o')
      ->fields('o')
      ->condition('id', $order_id)
      ->execute()
      ->fetchObject();

    if (!$order) {
      throw new NotFoundHttpException();
    }

    // When printing KOT/BOT mark as served (if not already).
    if (in_array($type, ['kot', 'bot']) && $order->status !== 'served' && $order->status !== 'completed') {
      $db->update('restaurant_order')
        ->fields(['status' => 'served'])
        ->condition('id', $order_id)
        ->execute();
      $order->status = 'served';
    }

    $items = json_decode($order->items, TRUE) ?: [];
    $service = \Drupal::service('restaurant_order.receipt_service');

    return $service->generateReceipt((object) [
      'id' => $order->id,
      'table_no' => $order->table_no,
      'items' => $items,
      'status' => $order->status,
      'is_aggregated' => FALSE,
      'session_id' => $order->session_id ?? NULL,
    ], $type);
  }

  /**
   * Print aggregated bill for a table + session.
   *
   * This fetches only non-completed orders for the specified session and marks
   * them completed after printing (closing the session).
   */
  public function printBillByTable($table_no, $session_id) {
    $db = \Drupal::database();
    $bill_exists = \Drupal::database()->select('restaurant_bill', 'b')
      ->fields('b', ['id'])
      ->condition('b.table_no', $table_no)
      ->condition('b.session_id', $session_id)
      ->execute()
      ->fetchField();

    if ($bill_exists) {
      \Drupal::messenger()->addMessage(t('Showing previously inserted bill.'));
    }

    // Only fetch orders that are not already completed.
    $orders = $db->select('restaurant_order', 'o')
      ->fields('o', ['id', 'items'])
      ->condition('table_no', $table_no)
      ->condition('session_id', $session_id)
      // ->condition('status', 'completed', '<>')
      ->execute()
      ->fetchAll();

    $aggregated_items = [];
    $ids = [];

    foreach ($orders as $order) {
      $ids[] = $order->id;
      $items = json_decode($order->items, TRUE) ?: [];
      foreach ($items as $item) {
        // Use combined key by id+price to safely aggregate identical menu items.
        $key = $item['id'] . '::' . (isset($item['price']) ? $item['price'] : '0');
        if (!isset($aggregated_items[$key])) {
          $aggregated_items[$key] = [
            'id' => $item['id'],
            'name' => $item['name'],
            'qty' => 0,
            'price' => $item['price'],
            'type' => $item['type'] ?? '',
          ];
        }
        $aggregated_items[$key]['qty'] += $item['qty'];
      }
    }

    $items = array_values($aggregated_items);
    if (empty($items)) {
      $this->messenger()->addMessage($this->t('No active orders found for Table %table (Session %session).', [
        '%table' => $table_no,
        '%session' => $session_id,
      ]));
      return [
        '#markup' => $this->t('No active orders found for Table %table (Session %session).', [
          '%table' => $table_no,
          '%session' => $session_id,
        ]),
      ];
    }

    // Mark all orders in this session as completed (closing the session).
    if (!empty($ids)) {
      $db->update('restaurant_order')
        ->fields(['status' => 'completed'])
        ->condition('id', $ids, 'IN')
        ->execute();
    }

    $order_obj = (object) [
      'id' => 'Aggregated for Table ' . $table_no . ' (S' . $session_id . ')',
      'table_no' => $table_no,
      'items' => $items,
      'status' => 'completed',
      'is_aggregated' => TRUE,
      'session_id' => $session_id,
    ];

    $service = \Drupal::service('restaurant_order.receipt_service');
    return $service->generateReceipt($order_obj, 'aggregated_bill');
  }

  /**
   * Update order status.
   */
  public function updateStatus($order_id, $status) {
    $valid_statuses = ['pending', 'served', 'completed', 'cancelled'];
    if (!in_array($status, $valid_statuses)) {
      $this->messenger()->addError($this->t('Invalid status.'));
      return new RedirectResponse(Url::fromRoute('restaurant_order.list_orders')->toString());
    }

    \Drupal::database()->update('restaurant_order')
      ->fields(['status' => $status])
      ->condition('id', $order_id)
      ->execute();

    $this->messenger()->addStatus($this->t('Order status updated to %status.', ['%status' => ucfirst($status)]));
    return new RedirectResponse(Url::fromRoute('restaurant_order.list_orders')->toString());
  }

  /**
   *
   */
  public function viewOrder($order_id) {
    $order = \Drupal::database()->select('restaurant_order', 'o')
      ->fields('o')
      ->condition('id', $order_id)
      ->execute()
      ->fetchObject();

    if (!$order) {
      throw new NotFoundHttpException();
    }

    $items = json_decode($order->items, TRUE) ?: [];

    $rows = [];
    foreach ($items as $item) {
      $rows[] = [
        $item['name'],
        $item['qty'],
        '₹' . number_format($item['price'], 2),
        ucfirst($item['type']),
        '₹' . number_format($item['qty'] * $item['price'], 2),
      ];
    }

    $actions = [
      '#type' => 'container',
      '#attributes' => ['class' => ['order-actions']],
    ];
    // Delete button also only if not completed.
    $delete_url = Url::fromRoute('restaurant_order.delete_order', ['order_id' => $order_id]);
    $actions['delete_button'] = [
      '#type' => 'link',
      '#title' => $this->t('Delete Order'),
      '#url' => $delete_url,
      '#attributes' => [
        'class' => ['button', 'button--danger'],
        'onclick' => ['return confirm("Are you sure you want to delete this order?");'],
      ],
    ];

    // Show "Repeat Order" only if not completed.
    if ($order->status !== 'completed') {
      $repeat_url = Url::fromRoute('restaurant_order.repeat_order', ['order_id' => $order_id]);
      $actions['repeat_button'] = [
        '#type' => 'link',
        '#title' => $this->t('Repeat Order'),
        '#url' => $repeat_url,
        '#attributes' => ['class' => ['button', 'button--primary']],
      ];
    }

    return [
      'wrapper' => [
        '#type' => 'container',
        '#attributes' => ['class' => ['order-view-wrapper']],
        'order_table' => [
          '#type' => 'table',
          '#header' => [
            $this->t('Item'),
            $this->t('Quantity'),
            $this->t('Unit Price'),
            $this->t('Type'),
            $this->t('Total'),
          ],
          '#rows' => $rows,
        ],
        'actions' => $actions,
      ],
      '#attached' => [
        'library' => ['restaurant_order/order_styles'],
      ],
    ];
  }

  /**
   * Repeat an existing order by pre-filling form with table number and session.
   * No items are carried forward to ensure KOT/BOT only prints new ones,
   * but repeat is considered part of same session.
   */
  public function repeatOrder($order_id) {
    // Load original order.
    $order = \Drupal::database()->select('restaurant_order', 'o')
      ->fields('o')
      ->condition('id', $order_id)
      ->execute()
      ->fetchObject();

    if (!$order) {
      throw new NotFoundHttpException();
    }

    // Determine session to reuse.
    $session_id = $order->session_id ?? $this->getActiveSessionForTable($order->table_no);

    // Pre-fill form with table number and session id.
    $default_values = [
      'table_no' => $order->table_no,
      'session_id' => $session_id,
      'parent_order_id' => $order->id,
    ];

    return \Drupal::formBuilder()->getForm('Drupal\\restaurant_order\\Form\\OrderForm', $default_values);
  }

  /**
   *
   */
  public function deleteOrder($order_id) {
    $db = \Drupal::database();

    $order = $db->select('restaurant_order', 'o')
      ->fields('o')
      ->condition('id', $order_id)
      ->execute()
      ->fetchObject();

    if (!$order) {
      throw new NotFoundHttpException('Order not found.');
    }

    // Delete the order.
    $db->delete('restaurant_order')
      ->condition('id', $order_id)
      ->execute();

    \Drupal::messenger()->addStatus($this->t('Order #%id has been deleted.', ['%id' => $order_id]));

    // Redirect to order list page.
    return new RedirectResponse(Url::fromRoute('restaurant_order.list_orders')->toString());
  }

  /**
   * Get an active session id for a table or return next available session id.
   *
   * - If there's an active (non-completed) session, return its session_id.
   * - Otherwise, return last_session_id + 1 (or 1 if none).
   */
  private function getActiveSessionForTable($table_no) {
    $db = \Drupal::database();

    // First, try to find an active session (session with any non-completed order).
    $active_session = $db->select('restaurant_order', 'o')
      ->fields('o', ['session_id'])
      ->condition('o.table_no', $table_no)
      ->condition('o.status', 'completed', '<>')
      ->orderBy('o.session_id', 'DESC')
      ->range(0, 1)
      ->execute()
      ->fetchField();

    if ($active_session) {
      return (int) $active_session;
    }

    // No active session — find last session_id and increment.
    $last_session = $db->select('restaurant_order', 'o')
      ->fields('o', ['session_id'])
      ->condition('o.table_no', $table_no)
      ->orderBy('o.session_id', 'DESC')
      ->range(0, 1)
      ->execute()
      ->fetchField();

    if ($last_session) {
      return (int) $last_session + 1;
    }

    // No previous session — start at 1.
    return 1;
  }

  /**
   *
   */
  public function aggregateBillDownload($table_no, $session_id) {
    $db = \Drupal::database();

    // Get archived bill if exists.
    $bill = $db->select('restaurant_bill', 'b')
      ->fields('b')
      ->condition('table_no', $table_no)
      ->condition('session_id', $session_id)
      ->execute()
      ->fetchObject();

    if (!$bill) {
      $this->messenger()->addError($this->t('No bill found for Table @table (Session @session).', [
        '@table' => $table_no,
        '@session' => $session_id,
      ]));
      return $this->redirect('restaurant_order.past_bills');
    }

    // Reuse receipt service for PDF.
    $service = \Drupal::service('restaurant_order.receipt_service');

    $order_obj = (object) [
      'id' => 'Bill ' . $bill->id,
      'table_no' => $bill->table_no,
      'items' => json_decode($bill->items, TRUE),
      'status' => 'completed',
      'is_aggregated' => TRUE,
    ];

    // Pass flag for download.
    return $service->generateReceipt($order_obj, 'aggregated_bill', TRUE);
  }

  /**
   * Helper: Generate bill PDF response.
   */
  private function generateBillPdf($table_no, $session_id, array $items, $grand_total, $archived = FALSE) {
    $html = '<h2>Aggregate Bill - Table ' . $table_no . ' (Session ' . $session_id . ')</h2>';
    if ($archived) {
      $html .= '<p><em>(From archive)</em></p>';
    }
    $html .= '<table border="1" cellspacing="0" cellpadding="5">';
    $html .= '<tr><th>Item</th><th>Qty</th><th>Price</th><th>Total</th></tr>';

    foreach ($items as $item) {
      $line_total = $item['qty'] * $item['price'];
      $html .= '<tr><td>' . $item['name'] . '</td><td>' . $item['qty'] . '</td><td>₹' .
      number_format($item['price'], 2) . '</td><td>₹' .
      number_format($line_total, 2) . '</td></tr>';
    }

    $html .= '<tr><td colspan="3"><strong>Grand Total</strong></td><td><strong>₹' .
    number_format($grand_total, 2) . '</strong></td></tr>';
    $html .= '</table>';

    $dompdf = new Dompdf();
    $dompdf->loadHtml($html);
    $dompdf->setPaper('A4', 'portrait');
    $dompdf->render();

    $response = new Response($dompdf->output());
    $response->headers->set('Content-Type', 'application/pdf');
    $response->headers->set('Content-Disposition', 'attachment; filename="aggregate_bill_table_' . $table_no . '_S' . $session_id . '.pdf"');
    return $response;
  }

  /**
   *
   */
  public function pastBills() {
    $header = [
    ['data' => $this->t('ID')],
    ['data' => $this->t('Table No')],
    ['data' => $this->t('Session ID')],
    ['data' => $this->t('Grand Total')],
    ['data' => $this->t('Created')],
    ['data' => $this->t('Download')],
    ];

    $rows = [];

    $bills = \Drupal::database()->select('restaurant_bill', 'b')
      ->fields('b')
      ->orderBy('created', 'DESC')
      ->execute()
      ->fetchAll();

    foreach ($bills as $bill) {
      $download_url = Url::fromRoute('restaurant_order.aggregate_bill_download', [
        'table_no' => $bill->table_no,
        'session_id' => $bill->session_id,
      ]);

      $download_link = Link::fromTextAndUrl($this->t('Download'), $download_url)->toString();

      $rows[] = [
        $bill->id,
        $bill->table_no,
        $bill->session_id,
        '₹' . number_format($bill->grand_total, 2),
        \Drupal::service('date.formatter')->format($bill->created, 'short'),
        $download_link,
      ];
    }

    return [
      '#type' => 'table',
      '#header' => $header,
      '#rows' => $rows,
      '#attributes' => [
        'class' => ['table', 'table-striped', 'table-hover', 'table-sm'],
      ],
      '#empty' => $this->t('No past bills found.'),
      '#cache' => ['max-age' => 0],
    ];
  }

  /**
   *
   */
  public function printAggregateBill($table_no, $session_id) {
    $connection = \Drupal::database();

    // Fetch all pending orders for this table + session.
    $orders = $connection->select('restaurant_order', 'o')
      ->fields('o')
      ->condition('o.table_no', $table_no)
      ->condition('o.session_id', $session_id)
      ->condition('o.status', 'completed', '<>')
      ->execute()
      ->fetchAll();

    if (empty($orders)) {
      drupal_set_message($this->t('No active orders for this table.'), 'warning');
      return $this->redirect('<front>');
    }

    // Calculate grand total.
    $grand_total = 0;
    $all_items = [];
    foreach ($orders as $order) {
      $items = json_decode($order->items, TRUE);
      foreach ($items as $item) {
        $grand_total += $item['price'] * $item['quantity'];
        $all_items[] = $item;
      }
    }

    // ✅ Insert aggregated bill record into restaurant_bill
    $bill_id = $connection->insert('restaurant_bill')
      ->fields([
        'table_no' => $table_no,
        'session_id' => $session_id,
        'grand_total' => $grand_total,
        'created' => \Drupal::time()->getCurrentTime(),
      ])
      ->execute();

    // Mark all included orders as completed.
    $connection->update('restaurant_order')
      ->fields(['status' => 'completed'])
      ->condition('table_no', $table_no)
      ->condition('session_id', $session_id)
      ->execute();

    // Generate PDF using your ReceiptService.
    $order = (object) [
      'id' => $bill_id,
      'table_no' => $table_no,
      'items' => $all_items,
    ];

    return $this->receiptService->generateReceipt($order, 'aggregated_bill');
  }

  /**
   *
   */
  public function aggregatedBill($order_id) {
    $order = $this->orderStorage->load($order_id);

    // Generate the inline PDF response.
    $pdf_response = $this->receiptService->generateReceipt($order, 'aggregated_bill');

    // Convert PDF content to base64 to allow auto-download without a second request.
    $pdf_base64 = base64_encode($pdf_response->getContent());
    $filename = "receipt_aggregated_bill_{$order->id}.pdf";

    return [
      '#type' => 'markup',
      '#markup' => '
      <iframe src="data:application/pdf;base64,' . $pdf_base64 . '" 
              style="width:100%;height:90vh;border:none;"></iframe>
      <script>
        // Auto-download the PDF
        const link = document.createElement("a");
        link.href = "data:application/pdf;base64,' . $pdf_base64 . '";
        link.download = "' . $filename . '";
        document.body.appendChild(link);
        link.click();
        document.body.removeChild(link);
      </script>
    ',
    ];
  }

  /**
   *
   */
  public function insertBill($table_no, $session_id) {
    $connection = \Drupal::database();

    // Fetch all orders for this table + session.
    $orders = $connection->select('restaurant_order', 'o')
      ->fields('o', ['id', 'items'])
      ->condition('o.table_no', $table_no)
      ->condition('o.session_id', $session_id)
      ->execute()
      ->fetchAll();

    if (empty($orders)) {
      \Drupal::messenger()->addWarning(t('No active orders found for Table @table (Session @session).', [
        '@table' => $table_no,
        '@session' => $session_id,
      ]));
      return $this->redirect('restaurant_order.list_orders');
    }

    $subtotal = 0;
    $gst_total = 0;
    $all_items = [];

    foreach ($orders as $order) {
      $items = json_decode($order->items, TRUE) ?: [];
      foreach ($items as $it) {
        $price = isset($it['price']) ? (float) $it['price'] : 0;
        $qty = isset($it['qty']) ? (int) $it['qty'] : 0;
        $item_total = $price * $qty;
        // 5% GST per item
        $item_gst = round($item_total * 0.05, 2);
        $item_grand = $item_total + $item_gst;

        $subtotal += $item_total;
        $gst_total += $item_gst;

        $key = strtolower($it['name']);
        if (!isset($all_items[$key])) {
          $all_items[$key] = [
            'name' => $it['name'],
            'qty' => $qty,
            'price' => $price,
            'subtotal' => $item_total,
            'gst' => $item_gst,
            'total' => $item_grand,
          ];
        }
        else {
          $all_items[$key]['qty'] += $qty;
          $all_items[$key]['subtotal'] += $item_total;
          $all_items[$key]['gst'] += $item_gst;
          $all_items[$key]['total'] += $item_grand;
        }
      }
    }

    $grand_total = $subtotal + $gst_total;
    $all_items_json = json_encode(array_values($all_items));

    // Check if bill already exists for table-session.
    $bill_id = $connection->select('restaurant_bill', 'b')
      ->fields('b', ['id'])
      ->condition('b.table_no', $table_no)
      ->condition('b.session_id', $session_id)
      ->execute()
      ->fetchField();

    if ($bill_id) {
      // Update existing bill.
      $connection->update('restaurant_bill')
        ->fields([
          'items' => $all_items_json,
          'subtotal' => $subtotal,
          'gst_total' => $gst_total,
          'grand_total' => $grand_total,
          'updated' => \Drupal::time()->getRequestTime(),
        ])
        ->condition('id', $bill_id)
        ->execute();

      \Drupal::messenger()->addStatus(t('Updated bill for Table @table (Session @session).', [
        '@table' => $table_no,
        '@session' => $session_id,
      ]));
    }
    else {
      // Insert new bill.
      $connection->insert('restaurant_bill')
        ->fields([
          'table_no' => $table_no,
          'session_id' => $session_id,
          'items' => $all_items_json,
          'subtotal' => $subtotal,
          'gst_total' => $gst_total,
          'grand_total' => $grand_total,
          'created' => \Drupal::time()->getRequestTime(),
        ])
        ->execute();

      \Drupal::messenger()->addStatus(t('Inserted new bill for Table @table (Session @session).', [
        '@table' => $table_no,
        '@session' => $session_id,
      ]));
    }

    return $this->redirect('restaurant_order.list_orders');
  }

}
