<?php

namespace Drupal\restaurant_order\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;

/**
 *
 */
class OrderForm extends FormBase {

  /**
   *
   */
  public function getFormId() {
    return 'restaurant_order_single_item_form';
  }

  /**
   *
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $form['#cache'] = ['max-age' => 0];
    $form['#tree'] = TRUE;

    $defaults = $form_state->getBuildInfo()['args'][0] ?? [];

    // Initialize added items if not already set.
    if ($form_state->get('added_items') === NULL) {
      $form_state->set('added_items', []);
    }

    // Table number field.
    $form['table_no'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Table Number'),
      '#required' => TRUE,
      '#default_value' => $defaults['table_no'] ?? '',
    ];

    // Hidden field to track parent order if repeating.
    if (!empty($defaults['parent_order_id'])) {
      $form['parent_order_id'] = [
        '#type' => 'hidden',
        '#value' => $defaults['parent_order_id'],
      ];
    }

    // Menu item input.
    $form['item_input'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Menu Item'),
      '#autocomplete_route_name' => 'restaurant_order.autocomplete_items',
      '#attributes' => ['autocomplete' => 'off'],
    ];

    // Quantity field.
    $form['item_qty'] = [
      '#type' => 'number',
      '#title' => $this->t('Quantity'),
      '#min' => 1,
      '#default_value' => 1,
    ];

    // Add item button.
    $form['add_item'] = [
      '#type' => 'submit',
      '#value' => $this->t('Add Item'),
      '#submit' => ['::addItemSubmit'],
      '#ajax' => [
        'callback' => '::ajaxRefreshItemList',
        'wrapper' => 'added-items-wrapper',
      ],
    ];

    // Display added items.
    $form['added_items'] = [
      '#type' => 'table',
      '#header' => [
        $this->t('Item'),
        $this->t('Quantity'),
      ],
      '#empty' => $this->t('No items added yet.'),
      '#prefix' => '<div id="added-items-wrapper">',
      '#suffix' => '</div>',
      '#attributes' => [
        'class' => ['order-items-table', 'table', 'table-striped'],
      ],
    ];

    foreach ($form_state->get('added_items') as $entry) {
      $form['added_items'][] = [
        'item' => ['#markup' => $entry['name']],
        'qty' => ['#markup' => $entry['qty']],
      ];
    }

    // Submit button.
    $form['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Place Order'),
      '#submit' => ['::submitForm'],
    ];

    // Attach CSS/JS library if needed.
    $form['#attached']['library'][] = 'restaurant_order/styles';

    return $form;
  }

  /**
   *
   */
  public function ajaxRefreshItemList(array &$form, FormStateInterface $form_state) {
    return $form['added_items'];
  }

  /**
   *
   */
  public function addItemSubmit(array &$form, FormStateInterface $form_state) {
    $item_name = trim($form_state->getValue('item_input'));
    $qty = (int) $form_state->getValue('item_qty');

    if ($item_name === '' || $qty <= 0) {
      return;
    }

    $item = \Drupal::database()->select('menu_item', 'm')
      ->fields('m', ['id', 'name', 'price', 'type'])
      ->condition('name', $item_name)
      ->execute()
      ->fetchObject();

    if ($item) {
      $added = $form_state->get('added_items');
      $added[] = [
        'id' => $item->id,
        'name' => $item->name,
        'qty' => $qty,
        'price' => $item->price,
        'type' => $item->type,
      ];
      $form_state->set('added_items', $added);

      // Reset fields.
      $form_state->setValue('item_input', '');
      $form_state->setValue('item_qty', 1);

      // Rebuild form.
      $form_state->setRebuild(TRUE);
    }
  }

  /**
   *
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $db = \Drupal::database();
    $table_no = $form_state->getValue('table_no');
    $items = $form_state->get('added_items');

    if (empty($items)) {
      $this->messenger()->addError($this->t('Please add at least one item.'));
      return;
    }

    // 1. Check if there is an active session for this table.
    $session_id = $db->select('restaurant_order', 'o')
      ->fields('o', ['session_id'])
      ->condition('table_no', $table_no)
      ->condition('status', 'completed', '<>')
      ->range(0, 1)
      ->orderBy('id', 'DESC')
      ->execute()
      ->fetchField();

    // 2. If no session, start a new one.
    if (!$session_id) {
      $session_id = time();
    }

    // 3. Insert the order.
    $fields = [
      'table_no' => $table_no,
      'items' => json_encode($items),
      'status' => 'new',
      'created' => \Drupal::time()->getRequestTime(),
      'session_id' => $session_id,
    ];

    if ($form_state->hasValue('parent_order_id')) {
      $fields['parent_order_id'] = $form_state->getValue('parent_order_id');
    }

    $db->insert('restaurant_order')->fields($fields)->execute();

    $this->messenger()->addStatus($this->t('Order placed successfully for Table @table (S@sid).', [
      '@table' => $table_no,
      '@sid' => $session_id,
    ]));

    $form_state->set('added_items', []);
    $form_state->setRebuild(TRUE);
  }

}
