<?php

namespace Drupal\restaurant_order\Service;

use Drupal\Core\Render\RendererInterface;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpFoundation\ResponseHeaderBag;

/**
 * Service to generate receipts (PDF) using TCPDF.
 */
class ReceiptService {
  protected $renderer;

  /**
   *
   */
  public function __construct(RendererInterface $renderer) {
    $this->renderer = $renderer;
  }

  /**
   * Generates and returns a Symfony Response streaming a PDF.
   * type: 'kot' | 'bot' | 'bill'
   */
  public function generateReceipt($order, $type = 'bill') {
    // Filter items depending on type.
    if ($type === 'kot') {
      $items = array_filter($order->items, function ($i) {
        return $i['type'] === 'food';
      });
    }
    elseif ($type === 'bot') {
      $items = array_filter($order->items, function ($i) {
        return $i['type'] === 'drink';
      });
    }
    else {
      $items = $order->items;
    }

    // Choose Twig template based on receipt type.
    $theme_map = [
      'kot' => 'receipt_kot',
      'bot' => 'receipt_bot',
      'bill' => 'receipt_bill',
    // NEW TEMPLATE.
      'aggregated_bill' => 'receipt_aggregated_bill',
    ];

    $theme = $theme_map[$type] ?? 'receipt_bill';

    // Render HTML content for the receipt.
    $render_array = [
      '#theme' => $theme,
      '#order' => (object) [
        'id' => $order->id,
        'table_no' => $order->table_no,
        'items' => $items,
      ],
      '#type' => $type,
    ];

    $html = $this->renderer->renderRoot($render_array);

    // Create new TCPDF object.
    $pdf = new \TCPDF(PDF_PAGE_ORIENTATION, PDF_UNIT, PDF_PAGE_FORMAT, TRUE, 'UTF-8', FALSE);

    // Set document information.
    $pdf->SetCreator('Drupal Restaurant Manager');
    $pdf->SetAuthor('Restaurant Manager');
    $pdf->SetTitle("Receipt {$type} #{$order->id}");
    $pdf->SetSubject('Receipt PDF');

    // Remove default header/footer.
    $pdf->setPrintHeader(FALSE);
    $pdf->setPrintFooter(FALSE);

    // Set margins.
    $pdf->SetMargins(10, 10, 10, TRUE);

    // Add a page.
    $pdf->AddPage();

    // Write the HTML content.
    $pdf->writeHTML($html, TRUE, FALSE, TRUE, FALSE, '');

    // Close and output PDF document as a string.
    // 'S' returns as string.
    $pdf_content = $pdf->Output('', 'S');

    // Create response with PDF content.
    $response = new Response($pdf_content);
    $disposition = $response->headers->makeDisposition(
      ResponseHeaderBag::DISPOSITION_INLINE,
      "receipt_{$type}_{$order->id}.pdf"
    );

    $response->headers->set('Content-Type', 'application/pdf');
    $response->headers->set('Content-Disposition', $disposition);

    return $response;
  }

}
