<?php

namespace Drupal\retroactive_workflow_state\Plugin\BatchPlugin;

use Drupal\batch_plugin\Attribute\BatchPlugin;
use Drupal\batch_plugin\BatchPluginBase;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\node\Entity\Node;

/**
 * Plugin implementation of the batch_plugin.
 */
#[BatchPlugin(
  id: "retroactive_workflow_state_batch_plugin",
  label: new TranslatableMarkup("Retroactive workflow state batch plugin"),
  description: new TranslatableMarkup("Give all existing content in content types configured to use a content moderation workflow a draft or published state (based on their unpublished or published status)."),
)]
class RetroactiveWorkflowStateBatchPlugin extends BatchPluginBase {

  /**
   * {@inheritDoc}
   */
  public function setupOperations(): void {
    $workflows = \Drupal::service('config.factory')->listAll('workflows.workflow.');
    $types = [];
    foreach ($workflows as $workflow_id) {
      $types = array_merge_recursive($types, \Drupal::config($workflow_id)->get("type_settings.entity_types"));
    }


    // Identify all nodes without any moderation state in this workflow.
    $query = \Drupal::database()->query("SELECT nid FROM {node_field_data} n LEFT JOIN {content_moderation_state_field_data} c ON n.nid = c.content_entity_id AND c.content_entity_type_id = 'node' WHERE type IN (:types[]) AND c.moderation_state IS NULL", [':types[]' => $types['node']]);

    // Fetch nids.
    $this->operations = $query->fetchCol();

    // Below should work to do the same as the above much more easily but
    // cannot until https://www.drupal.org/project/drupal/issues/3025164
    /*
    $query = \Drupal::entityQuery('node')
      ->accessCheck(FALSE)
      ->condition('moderation_state', NULL, 'IS NULL');
      if ($types['node']) {
        // Nodes use 'type' for their bundle key.
        $query->condition('type', $types['node'], 'IN');
      }
    $this->operations = $query->execute();
    */
  }

  /**
   * {@inheritDoc}
   */
  public function processOperation($payload, &$context): void {
    $node = Node::load($payload);
    $node->save();
    $context['results']['nodes'][$node->id()] = $node->id();
  }

  /**
   * {@inheritDoc}
   */
  public function finished(bool $success, array $results, array $operations): void {
    parent::finished($success, $results, $operations);
    $context = [
      '@count' => isset($results['nodes']) ? count($results['nodes']) : 0,
    ];
    \Drupal::messenger()->addMessage($this->t('@count nodes saved to set workflow state.', $context));
  }

}