<?php

namespace Drupal\reviewer_notes\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Url;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;

/**
 * Controller for reviewer notes administrative functions.
 *
 * This controller provides administrative interfaces for managing reviewer
 * notes, including listing, filtering, and exporting functionality.
 */
class ReviewerNotesAdminController extends ControllerBase {

  /**
   * Displays the admin list page for reviewer notes.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The current request.
   *
   * @return array
   *   Render array for the admin list page.
   */
  public function list(Request $request) {
    $path_filter = trim((string) $request->query->get('path'));
    $tags_filter = trim((string) $request->query->get('tags'));

    $build = [];
    $build['filters'] = [
      '#type' => 'container',
      'form' => [
        '#type' => 'html_tag',
        '#tag' => 'form',
        '#attributes' => [
          'method' => 'get',
          'action' => Url::fromRoute('reviewer_notes.admin_list')->toString(),
          'class' => ['container-inline'],
        ],
        'path' => [
          '#type' => 'textfield',
          '#title' => $this->t('Path contains'),
          '#default_value' => $path_filter,
          '#value' => $path_filter,
          '#attributes' => ['name' => 'path'],
        ],
        'tags' => [
          '#type' => 'textfield',
          '#title' => $this->t('Tags (comma/space separated)'),
          '#default_value' => $tags_filter,
          '#value' => $tags_filter,
          '#attributes' => ['name' => 'tags'],
        ],
        'actions' => [
          '#type' => 'container',
          'apply' => [
            '#type' => 'html_tag',
            '#tag' => 'button',
            '#value' => $this->t('Filter'),
            '#attributes' => ['type' => 'submit', 'class' => ['button', 'button--primary']],
          ],
          'export' => [
            '#type' => 'link',
            '#title' => $this->t('Export CSV'),
            '#url' => Url::fromRoute(
              'reviewer_notes.admin_export',
              [],
              ['query' => ['path' => $path_filter, 'tags' => $tags_filter]]
            ),
            '#attributes' => ['class' => ['button']],
          ],
        ],
      ],
    ];

    // Ensure render cache varies by query args so values persist.
    $build['#cache'] = [
      'contexts' => ['url.query_args:path', 'url.query_args:tags'],
    ];

    $query = $this->database()->select('reviewer_notes', 'rn')
      ->fields('rn', ['id', 'uid', 'path', 'selector', 'note', 'tags', 'status', 'created', 'changed'])
      ->orderBy('created', 'DESC');
    // Add pagination: 30 per page.
    $pager = $query->extend('Drupal\\Core\\Database\\Query\\PagerSelectExtender')->limit(30);
    if ($path_filter !== '') {
      $pager->condition('path', '%' . $this->database()->escapeLike($path_filter) . '%', 'LIKE');
    }
    if ($tags_filter !== '') {
      // Simple LIKE match; advanced tokenization can be added later.
      $pager->condition('tags', '%' . $this->database()->escapeLike($tags_filter) . '%', 'LIKE');
    }
    $rows = $pager->execute()->fetchAll();

    $header = [
      $this->t('ID'),
      $this->t('Path'),
      $this->t('Note'),
      $this->t('Tags'),
      $this->t('Status'),
      $this->t('Created'),
      $this->t('Changed'),
    ];
    $table_rows = [];
    foreach ($rows as $r) {
      $table_rows[] = [
        (int) $r->id,
        $r->path,
        $r->note,
        $r->tags,
        $r->status,
        \Drupal::service('date.formatter')->format($r->created, 'short'),
        \Drupal::service('date.formatter')->format($r->changed, 'short'),
      ];
    }

    $build['table'] = [
      '#type' => 'table',
      '#header' => $header,
      '#rows' => $table_rows,
      '#empty' => $this->t('No notes found.'),
    ];

    // Add pager below the table.
    $build['pager'] = [
      '#type' => 'pager',
    ];

    return $build;
  }

  /**
   * Exports reviewer notes as CSV.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The current request.
   *
   * @return \Symfony\Component\HttpFoundation\Response
   *   CSV response for download.
   */
  public function exportCsv(Request $request): Response {
    $path_filter = trim((string) $request->query->get('path'));
    $tags_filter = trim((string) $request->query->get('tags'));

    $query = $this->database()->select('reviewer_notes', 'rn')
      ->fields('rn', ['id', 'uid', 'path', 'selector', 'note', 'tags', 'status', 'created', 'changed'])
      ->orderBy('created', 'DESC');
    if ($path_filter !== '') {
      $query->condition('path', '%' . $this->database()->escapeLike($path_filter) . '%', 'LIKE');
    }
    if ($tags_filter !== '') {
      $query->condition('tags', '%' . $this->database()->escapeLike($tags_filter) . '%', 'LIKE');
    }
    $rows = $query->execute()->fetchAll();

    $out = fopen('php://temp', 'w+');
    fputcsv($out, ['ID', 'Path', 'Note', 'Tags', 'Status', 'Created', 'Changed']);
    foreach ($rows as $r) {
      fputcsv($out, [
        (int) $r->id,
        $r->path,
        $r->note,
        $r->tags,
        $r->status,
        date('c', (int) $r->created),
        date('c', (int) $r->changed),
      ]);
    }
    rewind($out);
    $csv = stream_get_contents($out);
    fclose($out);

    $response = new Response($csv);
    $response->headers->set('Content-Type', 'text/csv; charset=UTF-8');
    $response->headers->set('Content-Disposition', 'attachment; filename="reviewer-notes.csv"');
    return $response;
  }

  /**
   * Gets the database connection.
   *
   * @return \Drupal\Core\Database\Connection
   *   The database connection.
   */
  private function database() {
    return \Drupal::database();
  }

}
