<?php

namespace Drupal\reviewer_notes\Controller;

use Drupal\Core\Controller\ControllerBase;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Drupal\Core\Database\Connection;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Component\Utility\Xss;

/**
 * Controller for handling reviewer notes API endpoints.
 *
 * This controller provides JSON responses for the reviewer notes overlay
 * and handles CRUD operations for notes via AJAX requests.
 */
class ReviewerNotesController extends ControllerBase {

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected Connection $database;

  /**
   * The Reviewer Notes service.
   *
   * @var \Drupal\reviewer_notes\ReviewerNotesService
   */
  protected $reviewerNotesService;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): self {
    $instance = new static();
    $instance->database = $container->get('database');
    $instance->reviewerNotesService = $container->get('reviewer_notes.service');
    return $instance;
  }

  /**
   * Returns a JSON response with all notes for a given path.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The current request.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   JSON response containing the notes.
   */
  public function listNotes(Request $request): JsonResponse {
    $path = $request->query->get('path');
    if (!$path) {
      $path = \Drupal::service('path.current')->getPath();
    }

    $query = $this->database->select('reviewer_notes', 'rn')
      ->fields('rn')
      ->condition('path', $path)
      ->orderBy('created', 'ASC');
    $result = $query->execute()->fetchAllAssoc('id');

    $notes = [];
    foreach ($result as $id => $row) {
      $notes[] = [
        'id' => (int) $row->id,
        'uid' => (int) $row->uid,
        'path' => $row->path,
        'selector' => $row->selector,
        'anchor' => $row->anchor_json ? json_decode($row->anchor_json, TRUE) : NULL,
        'note' => $row->note,
        'tags' => $row->tags,
        'status' => $row->status,
        'created' => (int) $row->created,
        'changed' => (int) $row->changed,
      ];
    }

    return new JsonResponse(['notes' => $notes]);
  }

  /**
   * Adds a new reviewer note.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The current request containing the note data.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   JSON response with the result.
   */
  public function add(Request $request): JsonResponse {
    $payload = json_decode($request->getContent(), TRUE) ?: [];
    $path = (string) ($payload['path'] ?? '');
    $selector = (string) ($payload['selector'] ?? '');
    $anchor = $payload['anchor'] ?? NULL;
    $note_text = (string) ($payload['note'] ?? '');
    $rawTags = (string) ($payload['tags'] ?? '');

    if ($path === '' || $note_text === '') {
      return new JsonResponse(['error' => 'Path and note are required.'], 400);
    }

    try {
      $id = $this->reviewerNotesService->addNote([
        'path' => $path,
        'selector' => $selector,
        'anchor' => $anchor,
        'note' => $note_text,
        'tags' => $rawTags,
      ]);
      return new JsonResponse(['id' => (int) $id]);
    }
    catch (\Exception $e) {
      return new JsonResponse(['error' => $e->getMessage()], 500);
    }
  }

  /**
   * Updates an existing reviewer note.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The current request containing the updated note data.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   JSON response with the result.
   */
  public function update(Request $request): JsonResponse {
    $payload = json_decode($request->getContent(), TRUE) ?: [];
    $id = (int) ($payload['id'] ?? 0);

    if ($id <= 0) {
      return new JsonResponse(['error' => 'Invalid id'], 400);
    }

    try {
      $this->reviewerNotesService->updateNote($payload);
      return new JsonResponse(['ok' => TRUE]);
    }
    catch (\Exception $e) {
      return new JsonResponse(['error' => $e->getMessage()], 500);
    }
  }

  /**
   * Deletes a reviewer note by ID.
   *
   * @param int $id
   *   The ID of the note to delete.
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The current request.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   JSON response with the result.
   */
  public function delete(int $id, Request $request): JsonResponse {
    if ($id <= 0) {
      return new JsonResponse(['error' => 'Invalid id'], 400);
    }
    try {
      $this->reviewerNotesService->deleteNote($id);
      return new JsonResponse(['ok' => TRUE]);
    }
    catch (\Exception $e) {
      return new JsonResponse(['error' => $e->getMessage()], 500);
    }
  }

  /**
   * Returns activity logs for a specific note.
   *
   * @param int $id
   *   The ID of the note.
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The current request.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   JSON response containing the activity logs.
   */
  public function logs(int $id, Request $request): JsonResponse {
    if ($id <= 0) {
      return new JsonResponse(['error' => 'Invalid id'], 400);
    }
    $query = $this->database->select('reviewer_notes_log', 'l')
      ->fields('l', ['id', 'note_id', 'uid', 'action', 'details', 'created'])
      ->condition('note_id', $id)
      ->orderBy('created', 'DESC');
    $query->leftJoin('users_field_data', 'u', 'u.uid = l.uid');
    $query->addField('u', 'name', 'user_name');

    $result = $query->execute()->fetchAll();

    $logs = [];
    foreach ($result as $row) {
      $uid = (int) $row->uid;
      $name = $row->user_name ?: ($uid === 0 ? 'Anonymous' : 'User');
      $logs[] = [
        'id' => (int) $row->id,
        'note_id' => (int) $row->note_id,
        'uid' => $uid,
        'user' => ['uid' => $uid, 'name' => $name],
        'action' => (string) $row->action,
        'details' => $row->details ? json_decode($row->details, TRUE) : NULL,
        'created' => (int) $row->created,
      ];
    }

    return new JsonResponse(['logs' => $logs]);
  }

  /**
   * Returns comments for a specific note.
   *
   * @param int $id
   *   The ID of the note.
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The current request.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   JSON response containing the comments.
   */
  public function listComments(int $id, Request $request): JsonResponse {
    if ($id <= 0) {
      return new JsonResponse(['error' => 'Invalid id'], 400);
    }
    $query = $this->database->select('reviewer_notes_comment', 'c')
      ->fields('c', ['id', 'note_id', 'uid', 'comment', 'created'])
      ->condition('note_id', $id)
      ->orderBy('created', 'ASC');
    $query->leftJoin('users_field_data', 'u', 'u.uid = c.uid');
    $query->addField('u', 'name', 'user_name');

    $result = $query->execute()->fetchAll();

    $comments = [];
    foreach ($result as $row) {
      $uid = (int) $row->uid;
      $name = $row->user_name ?: ($uid === 0 ? 'Anonymous' : 'User');
      $comments[] = [
        'id' => (int) $row->id,
        'note_id' => (int) $row->note_id,
        'uid' => $uid,
        'user' => ['uid' => $uid, 'name' => $name],
        'comment' => (string) $row->comment,
        'created' => (int) $row->created,
      ];
    }

    return new JsonResponse(['comments' => $comments]);
  }

  /**
   * Adds a comment to a specific note.
   *
   * @param int $id
   *   The ID of the note.
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The current request containing the comment data.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   JSON response with the result.
   */
  public function addComment(int $id, Request $request): JsonResponse {
    if ($id <= 0) {
      return new JsonResponse(['error' => 'Invalid id'], 400);
    }
    $payload = json_decode($request->getContent(), TRUE) ?: [];
    $comment = Xss::filter((string) ($payload['comment'] ?? ''));
    if ($comment === '') {
      return new JsonResponse(['error' => 'Comment is required'], 400);
    }
    $time = \Drupal::time()->getRequestTime();
    $cid = $this->database->insert('reviewer_notes_comment')->fields([
      'note_id' => $id,
      'uid' => (int) $this->currentUser()->id(),
      'comment' => $comment,
      'created' => $time,
    ])->execute();

    $this->reviewerNotesService->logActivity($id, 'comment', ['id' => (int) $cid, 'comment' => $comment]);

    return new JsonResponse(['id' => (int) $cid]);
  }

  /**
   * Normalizes tags string (deprecated - moved to service).
   *
   * @param string $raw
   *   The raw tags string.
   *
   * @throws \Exception
   *   Always throws an exception as this method has been moved to the service.
   */
  private function normalizeTags(string $raw) {
    // This function is now in the service, so we don't need it here.
    throw new \Exception('Normalize tags function should be called from the service.');
  }

  /**
   * Logs activity for a note (deprecated - moved to service).
   *
   * @param int $noteId
   *   The note ID.
   * @param string $action
   *   The action performed.
   * @param array $details
   *   Additional details about the action.
   *
   * @throws \Exception
   *   Always throws an exception as this method has been moved to the service.
   */
  private function logActivity(int $noteId, string $action, array $details) {
    // This function is now in the service, so we don't need it here.
    throw new \Exception('Log activity function should be called from the service.');
  }

}
