<?php

namespace Drupal\reviewer_notes\Access;

use Drupal\Core\Access\AccessCheckInterface;
use Drupal\Core\Access\AccessResult;
use Drupal\Core\Access\CsrfTokenGenerator;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Session\AccountInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\Routing\Route;

/**
 * Access check for CSRF token validation in reviewer notes.
 *
 * This class implements custom access checking for reviewer notes routes
 * that require CSRF token validation based on module configuration.
 */
class ReviewerNotesCsrfAccessCheck implements AccessCheckInterface {

  public function __construct(
    protected ConfigFactoryInterface $configFactory,
    protected CsrfTokenGenerator $csrfToken,
  ) {}

  /**
   * {@inheritdoc}
   */
  public function applies(Route $route) {
    return $route->getRequirement('_reviewer_notes_csrf') !== NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function access(Route $route, Request $request, AccountInterface $account): AccessResult {
    $config = $this->configFactory->get('reviewer_notes.settings');
    $enabled = (bool) $config->get('csrf_protect');
    if (!$enabled) {
      return AccessResult::allowed()->addCacheableDependency($config);
    }

    $method = strtoupper($request->getMethod());
    if (in_array($method, ['POST', 'PATCH', 'PUT', 'DELETE'], TRUE)) {
      $token = $request->headers->get('X-CSRF-Token');
      $is_valid = is_string($token) && $this->csrfToken->validate($token);
      return AccessResult::allowedIf($is_valid)
        ->addCacheableDependency($config)
        ->addCacheContexts(['headers:x-csrf-token']);
    }

    return AccessResult::allowed()->addCacheableDependency($config);
  }

}
