<?php

namespace Drupal\reviewer_notes\Form;

use Drupal\Core\Form\ConfirmFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;

/**
 * Provides a confirmation form for deleting all reviewer notes.
 *
 * This form displays the total count of notes that will be deleted and asks
 * for confirmation before proceeding with the bulk delete operation.
 */
class ReviewerNotesDeleteAllConfirmForm extends ConfirmFormBase {

  /**
   * No filters; this form deletes ALL notes.
   */
  public function getFormId() {
    return 'reviewer_notes_delete_all_confirm_form';
  }

  /**
   * {@inheritdoc}
   */
  public function getQuestion() {
    return $this->t('Are you sure you want to delete all notes matching the current filters?');
  }

  /**
   * {@inheritdoc}
   */
  public function getCancelUrl() {
    return Url::fromRoute('reviewer_notes.admin_manage');
  }

  /**
   * {@inheritdoc}
   */
  public function getConfirmText() {
    return $this->t('Delete all');
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    // Show how many total notes will be deleted.
    $count = (int) \Drupal::database()->select('reviewer_notes', 'rn')->countQuery()->execute()->fetchField();
    $form['info'] = [
      '#type' => 'item',
      '#title' => $this->t('Total notes'),
      '#markup' => $this->t('@count note(s) will be deleted.', ['@count' => $count]),
    ];
    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $db = \Drupal::database();
    $count = $db->delete('reviewer_notes')->execute();
    $this->messenger()->addStatus($this->t('Deleted @count note(s).', ['@count' => $count]));
    $form_state->setRedirectUrl($this->getCancelUrl());
  }

}
