<?php

namespace Drupal\reviewer_notes\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;

/**
 * Provides a form for managing reviewer notes in the admin interface.
 *
 * This form displays a table of all reviewer notes with filtering and bulk
 * operations capabilities.
 */
class ReviewerNotesManageForm extends FormBase {

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'reviewer_notes_manage_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $path = trim((string) $this->getRequest()->query->get('path'));
    $tags = trim((string) $this->getRequest()->query->get('tags'));

    $form['filters'] = [
      '#type' => 'container',
      '#weight' => -20,
    ];
    $form['filters']['path'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Path contains'),
      '#default_value' => $path,
    ];
    $form['filters']['tags'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Tags (comma/space separated)'),
      '#default_value' => $tags,
    ];

    $form['actions'] = ['#type' => 'actions', '#weight' => -10];
    $form['actions']['apply'] = [
      '#type' => 'submit',
      '#value' => $this->t('Filter'),
      '#submit' => ['::submitFilter'],
    ];
    $form['actions']['delete_selected'] = [
      '#type' => 'submit',
      '#value' => $this->t('Delete selected'),
      '#submit' => ['::submitBulkDelete'],
    ];
    $form['actions']['delete_all'] = [
      '#type' => 'link',
      '#title' => $this->t('Delete all notes'),
      '#url' => Url::fromRoute(
        'reviewer_notes.admin_delete_all_confirm',
        [],
        ['query' => ['path' => $path, 'tags' => $tags]]
      ),
      '#attributes' => ['class' => ['button', 'button--danger']],
      '#weight' => 5,
    ];

    // Table of notes with checkboxes.
    $header = [
      'select' => $this->t('Select'),
      'id' => $this->t('ID'),
      'path' => $this->t('Path'),
      'note' => $this->t('Note'),
      'tags' => $this->t('Tags'),
      'status' => $this->t('Status'),
      'ops' => $this->t('Operations'),
    ];

    $query = \Drupal::database()->select('reviewer_notes', 'rn')
      ->fields('rn', ['id', 'path', 'note', 'tags', 'status'])
      ->orderBy('id', 'DESC');
    $pager = $query->extend('Drupal\\Core\\Database\\Query\\PagerSelectExtender')->limit(30);
    if ($path !== '') {
      $pager->condition('path', '%' . \Drupal::database()->escapeLike($path) . '%', 'LIKE');
    }
    if ($tags !== '') {
      $pager->condition('tags', '%' . \Drupal::database()->escapeLike($tags) . '%', 'LIKE');
    }
    $rows = $pager->execute()->fetchAll();

    $form['notes'] = [
      '#type' => 'table',
      '#header' => $header,
      '#empty' => $this->t('No notes found.'),
      '#weight' => 0,
    ];
    foreach ($rows as $r) {
      $form['notes'][$r->id]['select'] = [
        '#type' => 'checkbox',
      ];
      $form['notes'][$r->id]['id'] = ['#markup' => (int) $r->id];
      $form['notes'][$r->id]['path'] = ['#markup' => $r->path];
      $form['notes'][$r->id]['note'] = ['#markup' => $r->note];
      $form['notes'][$r->id]['tags'] = ['#markup' => $r->tags];
      $form['notes'][$r->id]['status'] = ['#markup' => $r->status];
      $form['notes'][$r->id]['ops'] = [
        '#type' => 'link',
        '#title' => $this->t('Delete'),
        '#url' => Url::fromRoute('reviewer_notes.admin_delete_confirm', ['id' => (int) $r->id]),
        '#attributes' => ['class' => ['button']],
      ];
    }

    $form['pager'] = [
      '#type' => 'pager',
      '#weight' => 10,
    ];

    $form['#cache'] = [
      'contexts' => ['url.query_args:path', 'url.query_args:tags'],
    ];

    return $form;
  }

  /**
   * Submit handler for the filter form.
   *
   * @param array $form
   *   An associative array containing the structure of the form.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   */
  public function submitFilter(array &$form, FormStateInterface $form_state) {
    $path = (string) $form_state->getValue('path');
    $tags = (string) $form_state->getValue('tags');
    $form_state->setRedirectUrl(
      Url::fromRoute(
        'reviewer_notes.admin_manage',
        [],
        ['query' => ['path' => $path, 'tags' => $tags]]
      )
    );
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    // No default submit. Specific actions are handled by submitFilter() and submitBulkDelete().
  }

  /**
   * Submit handler for bulk delete action.
   *
   * @param array $form
   *   An associative array containing the structure of the form.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   */
  public function submitBulkDelete(array &$form, FormStateInterface $form_state) {
    $selected = [];
    foreach ((array) $form_state->getValue('notes') as $id => $row) {
      if (!empty($row['select'])) {
        $selected[] = (int) $id;
      }
    }
    if (empty($selected)) {
      $this->messenger()->addWarning($this->t('No notes selected.'));
      return;
    }
    $form_state->setRedirect(
      'reviewer_notes.admin_bulk_delete_confirm',
      [],
      ['query' => ['ids' => implode(',', $selected)]]
    );
  }

}
