(function (Drupal, drupalSettings, once) {
  const settings = drupalSettings.reviewerNotes || {};
  const endpoints = settings.endpoints || {};
  const currentPath = settings.path || window.location.pathname;
  const canAdd = !!settings.canAdd;
  const canEdit = !!settings.canEdit;
  const canDelete = !!settings.canDelete;
  const csrfProtect = !!settings.csrfProtect;

  // Highlight-all state across the overlay lifecycle.
  let highlightAllActive = false;
  /** @type {Set<Element>} */
  const highlightedElements = new Set();

  let isPicking = false;
  let currentHoverEl = null;
  let detachPickerHandlers = null;

  function escapeHtml(str) {
    const div = document.createElement('div');
    div.textContent = String(str ?? '');
    return div.innerHTML;
  }

  async function requestJSON(url, options) {
    const opts = {
      credentials: 'same-origin',
      ...(options || {}),
    };
    const method = (opts.method || 'GET').toUpperCase();
    if (csrfProtect && method !== 'GET' && endpoints.csrfToken) {
      try {
        const tokenRes = await fetch(endpoints.csrfToken, {
          credentials: 'same-origin',
        });
        const token = await tokenRes.text();
        opts.headers = { 'X-CSRF-Token': token, ...(opts.headers || {}) };
      } catch (e) {
        // proceed without token if fetch fails
      }
    }
    const res = await fetch(url, opts);
    let data = null;
    try {
      data = await res.json();
    } catch (e) {
      // ignore parse errors
    }
    if (!res.ok || (data && data.error)) {
      const message =
        (data && data.error) || res.statusText || 'Request failed';
      throw new Error(message);
    }
    return data || {};
  }

  async function getNotes() {
    const url = `${endpoints.list}?path=${encodeURIComponent(currentPath)}`;
    const res = await fetch(url, { credentials: 'same-origin' });
    const json = await res.json();
    return json.notes || [];
  }

  async function getLogs(noteId) {
    const logsBase = (endpoints.logs || '').replace(/\/?0\/?logs$/, '');
    const url = `${logsBase}/${noteId}/logs`;
    const res = await fetch(url, { credentials: 'same-origin' });
    const json = await res.json();
    return json.logs || [];
  }

  async function listComments(noteId) {
    const base = (endpoints.commentsList || '').replace(/\/?0\/?comments$/, '');
    const url = `${base}/${noteId}/comments`;
    return requestJSON(url);
  }

  function scrollToSelectorOrY(selector, y) {
    let scrolled = false;
    if (selector) {
      const target = document.querySelector(selector);
      if (target) {
        target.scrollIntoView({ behavior: 'smooth', block: 'center' });
        target.classList.add('rn-highlight');
        setTimeout(() => target.classList.remove('rn-highlight'), 1500);
        scrolled = true;
      }
    }
    if (!scrolled && typeof y === 'number' && !Number.isNaN(y)) {
      const top = Math.max(0, y - Math.floor(window.innerHeight / 2));
      window.scrollTo({ top, behavior: 'smooth' });
      scrolled = true;
    }
    return scrolled;
  }

  async function updateNote(payload) {
    return requestJSON(endpoints.update, {
      method: 'POST',
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify(payload),
    });
  }

  async function deleteNote(id) {
    const deleteBase = (endpoints.delete || '').replace(/\/?0\/?delete$/, '');
    const url = `${deleteBase}/${id}/delete`;
    return requestJSON(url, { method: 'POST' });
  }

  function disablePicker() {
    if (!isPicking) return;
    isPicking = false;
    document.body.classList.remove('rn-picking');
    if (currentHoverEl) currentHoverEl.classList.remove('rn-pick-highlight');
    currentHoverEl = null;
    if (detachPickerHandlers) detachPickerHandlers();
  }

  function normalizeTarget(e) {
    let el = e.target;
    if (!(el instanceof Element)) {
      el = el && el.parentElement ? el.parentElement : null;
    }
    return el;
  }

  function userLabel(obj) {
    let uid = 0;
    if (obj && typeof obj.uid === 'number') {
      uid = obj.uid;
    } else if (obj && obj.uid) {
      uid = parseInt(obj.uid, 10);
    }

    let name = 'User';
    if (obj && obj.name) {
      name = String(obj.name);
    } else if (uid === 0) {
      name = 'Anonymous';
    }

    return `${escapeHtml(name)}(${uid})`;
  }

  function renderLogs(logs) {
    return (logs || [])
      .map((l) => {
        const when = new Date((l.created || 0) * 1000).toLocaleString();
        const action = escapeHtml(l.action || '');

        let user = '';
        if (l.user) {
          user = userLabel(l.user);
        } else if (typeof l.uid !== 'undefined') {
          user = `User(${l.uid})`;
        }

        let detailsText = '';
        if (l.action === 'comment' && l.details && l.details.comment) {
          detailsText = escapeHtml(l.details.comment);
        } else if (l.details) {
          detailsText = escapeHtml(JSON.stringify(l.details));
        }

        return `<div class="rn-log">
        <span class="rn-log-message">
          <span class="rn-log-when">${when}</span> —
          <span class="rn-log-action">${action}</span>
          <span class="rn-log-user">${user}</span>
          <span class="rn-log-details">${detailsText}</span>
        </span>
      </div>`;
      })
      .join('');
  }

  function renderComments(comments) {
    return (comments || [])
      .map((c) => {
        const when = new Date((c.created || 0) * 1000).toLocaleString();

        let user = '';
        if (c.user) {
          user = userLabel(c.user);
        } else if (typeof c.uid !== 'undefined') {
          user = `User(${c.uid})`;
        }

        return `<div class="rn-comment">
        <span class="rn-comment-when">${when}</span>
        <span class="rn-comment-user">${user}</span>: ${escapeHtml(c.comment || '')}
      </div>`;
      })
      .join('');
  }

  function createOverlay() {
    const container = document.createElement('div');
    container.className = 'rn-overlay rn-collapsed'; // Add rn-collapsed class by default.
    container.innerHTML = `
      <div class="rn-toolbar">
        <div class="rn-left">
          <strong>Reviewer Notes</strong>
          <span class="rn-filter-wrap">
            <input type="text" placeholder="Filter by tag (comma/space separated)" class="rn-filter" />
            <button type="button" class="rn-filter-clear" aria-label="Clear filter">×</button>
          </span>
          <span class="rn-count" aria-live="polite">0/0</span>
          <span class="rn-count-status" aria-live="polite">open: 0 · resolved: 0</span>
        </div>
        <div class="rn-right">
          ${canAdd ? '<button class="rn-pick">Pick element</button><button class="rn-add">Quick note</button>' : ''}
          <button class="rn-highlight-all" style="display: none;">Highlight all</button>
          <button class="rn-toggle">Expand</button>
        </div>
      </div>
      <div class="rn-list"></div>
    `;
    document.body.appendChild(container);

    const toggle = container.querySelector('.rn-toggle');
    toggle.addEventListener('click', () => {
      container.classList.toggle('rn-collapsed');
      toggle.textContent = container.classList.contains('rn-collapsed')
        ? 'Expand'
        : 'Collapse';
    });

    return container;
  }

  function clearAllHighlights() {
    highlightedElements.forEach((el) => {
      try {
        el.classList.remove('rn-highlight');
      } catch (e) {
        // ignore errors
      }
    });
    highlightedElements.clear();
  }

  async function highlightAllAnnotations(notesList) {
    // Accept a preloaded notes list when available to avoid extra fetches.
    let notes = notesList;
    if (!Array.isArray(notes)) {
      try {
        notes = await getNotes();
      } catch (e) {
        notes = [];
      }
    }
    clearAllHighlights();
    const selectors = Array.from(
      new Set(
        (notes || [])
          .map((n) =>
            (n.selector || (n.anchor && n.anchor.selector) || '').trim(),
          )
          .filter(Boolean),
      ),
    );
    selectors.forEach((sel) => {
      try {
        const el = document.querySelector(sel);
        if (el) {
          el.classList.add('rn-highlight');
          highlightedElements.add(el);
        }
      } catch (e) {
        // ignore invalid selectors
      }
    });
  }

  function getUniqueSelector(el) {
    if (!(el instanceof Element)) return '';
    const parts = [];
    while (el && el.nodeType === 1 && el !== document.body) {
      let selector = el.nodeName.toLowerCase();
      if (el.id) {
        selector += `#${CSS.escape(el.id)}`;
        parts.unshift(selector);
        break;
      } else {
        let siblingIndex = 1;
        let sib = el.previousElementSibling;
        while (sib) {
          if (sib.nodeName.toLowerCase() === selector) siblingIndex++;
          sib = sib.previousElementSibling;
        }
        selector += `:nth-of-type(${siblingIndex})`;
      }
      parts.unshift(selector);
      el = el.parentElement;
    }
    return parts.join(' > ');
  }

  function serializeAnchorFromElement(element) {
    if (!element) return null;
    const rect = element.getBoundingClientRect();
    const path = getUniqueSelector(element);
    return {
      selector: path,
      rect: {
        x: rect.x,
        y: rect.y + window.scrollY,
        width: rect.width,
        height: rect.height,
      },
    };
  }

  function attachItemHandlers(item) {
    const id = parseInt(item.dataset.id, 10) || 0;
    if (!id) return;
    const statusSpan = item.querySelector('.rn-status');

    const onResolve = async (e) => {
      e.preventDefault();
      e.stopPropagation();
      if (!canEdit) return;
      const isResolved =
        item.classList.contains('rn-status-resolved') ||
        item.dataset.status === 'resolved';
      const newStatus = isResolved ? 'open' : 'resolved';
      item.dataset.status = newStatus;
      item.classList.remove('rn-status-open', 'rn-status-resolved');
      item.classList.add(`rn-status-${newStatus}`);
      if (statusSpan) statusSpan.textContent = newStatus;
      const btn = e.currentTarget;
      btn.textContent = newStatus === 'resolved' ? 'Re-open' : 'Resolve';
      try {
        await updateNote({ id, status: newStatus });
        const logsEl = item.querySelector('.rn-logs');
        if (logsEl && logsEl.hasAttribute('data-loaded')) {
          const logs = await getLogs(id);
          logsEl.innerHTML = renderLogs(logs);
        }
        // Only refresh if we need to update the list filtering
        // For status changes, we can update in place without full refresh
        const currentFilterEl = document.querySelector('.rn-filter');
        if (currentFilterEl && currentFilterEl.value) {
          window.dispatchEvent(new CustomEvent('rn-refresh'));
        }
      } catch (err) {
        alert(`Failed to update status: ${err.message}`);
      }
    };

    const onEdit = async (e) => {
      e.preventDefault();
      e.stopPropagation();
      if (!canEdit) return;
      const currentNoteElement = item.querySelector('.rn-note');
      // Extract only the text content, excluding the '#ID' span.
      const currentText = Array.from(currentNoteElement.childNodes)
        .filter((node) => node.nodeType === Node.TEXT_NODE)
        .map((node) => node.textContent)
        .join('')
        .trim();
      const currentTags = Array.from(item.querySelectorAll('.rn-tag'))
        .map((t) => t.textContent)
        .join(', ');
      const newNote = prompt('Update note text:', currentText);
      if (newNote == null) return;
      const trimmedNote = newNote.trim();
      if (trimmedNote === '') {
        alert('Oops! Looks like you forgot to add the note text.');
        return;
      }
      const newTags =
        prompt('Enter tags (comma-separated):', currentTags) || '';
      try {
        await updateNote({ id, note: trimmedNote, tags: newTags });
        const noteEl = item.querySelector('.rn-note');
        if (noteEl) noteEl.textContent = trimmedNote;
        const tagsEl = item.querySelector('.rn-tags');
        if (tagsEl) {
          const tagHtml = (newTags || '')
            .split(/[\s,]+/)
            .map((t) => t.trim())
            .filter(Boolean)
            .map((t) => `<span class="rn-tag">${escapeHtml(t)}</span>`)
            .join('');
          tagsEl.innerHTML = tagHtml;
        }
        const logsEl = item.querySelector('.rn-logs');
        if (logsEl && logsEl.hasAttribute('data-loaded')) {
          const logs = await getLogs(id);
          logsEl.innerHTML = renderLogs(logs);
        }
      } catch (err) {
        alert(`Failed to update note: ${err.message}`);
      }
    };

    const onDelete = async (e) => {
      e.preventDefault();
      e.stopPropagation();
      if (!canDelete) return;
      if (!window.confirm('Delete this note?')) return;
      try {
        await deleteNote(id);
        item.remove();
        window.dispatchEvent(new CustomEvent('rn-refresh'));
      } catch (err) {
        alert(`Failed to delete note: ${err.message}`);
      }
    };

    const onScroll = (e) => {
      e.preventDefault();
      e.stopPropagation();
      const selector = item.dataset.selector || '';
      const anchorY = item.dataset.anchorY
        ? parseFloat(item.dataset.anchorY)
        : NaN;
      scrollToSelectorOrY(selector, anchorY);
    };

    const onLogs = async (e) => {
      e.preventDefault();
      e.stopPropagation();
      const logsEl = item.querySelector('.rn-logs');
      const btn = e.currentTarget;
      if (!logsEl) return;
      if (!logsEl.hasAttribute('data-loaded')) {
        logsEl.textContent = 'Loading logs...';
        const logs = await getLogs(id);
        logsEl.innerHTML = renderLogs(logs);
        logsEl.setAttribute('data-loaded', '1');
      }
      logsEl.hidden = !logsEl.hidden;
      btn.textContent = logsEl.hidden ? 'Show logs' : 'Hide logs';
    };

    const onComments = async (e) => {
      e.preventDefault();
      e.stopPropagation();
      const wrap = item.querySelector('.rn-comments');
      const btn = e.currentTarget;
      if (!wrap) return;
      const list = wrap.querySelector('.rn-comments-list');
      if (!wrap.hasAttribute('data-loaded')) {
        list.textContent = 'Loading comments...';
        try {
          const { comments } = await listComments(id);
          list.innerHTML = renderComments(comments);
          wrap.setAttribute('data-loaded', '1');
        } catch (err) {
          list.textContent = `Failed to load comments: ${err.message}`;
        }
      }
      wrap.hidden = !wrap.hidden;
      btn.textContent = wrap.hidden ? 'Show comments' : 'Hide comments';
    };

    item
      .querySelector('[data-action="resolve"]')
      ?.addEventListener('click', onResolve);
    item
      .querySelector('[data-action="edit"]')
      ?.addEventListener('click', onEdit);
    item
      .querySelector('[data-action="delete"]')
      ?.addEventListener('click', onDelete);
    item
      .querySelector('[data-action="scroll"]')
      ?.addEventListener('click', onScroll);
    item
      .querySelector('[data-action="logs"]')
      ?.addEventListener('click', onLogs);
    item
      .querySelector('[data-action="comments"]')
      ?.addEventListener('click', onComments);
  }

  function buildNoteItem(note) {
    const div = document.createElement('div');
    div.className = `rn-item rn-status-${note.status}`;
    div.dataset.id = note.id;
    div.dataset.selector =
      note.selector || (note.anchor && note.anchor.selector) || '';
    div.dataset.status = (note.status || '').toLowerCase();
    const anchorY =
      note.anchor && note.anchor.rect && typeof note.anchor.rect.y === 'number'
        ? note.anchor.rect.y
        : '';
    if (anchorY !== '') {
      div.dataset.anchorY = String(anchorY);
    }
    const hasAnchor = !!div.dataset.selector;
    const tagHtml = (note.tags || '')
      .split(',')
      .map((t) => t.trim())
      .filter(Boolean)
      .map((t) => `<span class="rn-tag">${escapeHtml(t)}</span>`)
      .join('');
    const resolveLabel =
      (note.status || '').toLowerCase() === 'resolved' ? 'Re-open' : 'Resolve';
    div.innerHTML = `
      <div class="rn-item-header">
        <span class="rn-tags">${tagHtml}</span>
        <span class="rn-status">${escapeHtml(note.status)}</span>
      </div>
      <div class="rn-id">#${note.id}</div><div class="rn-note"> ${escapeHtml(note.note)}</div>
      <div class="rn-actions">
        ${canEdit ? `<button data-action="resolve">${resolveLabel}</button>` : ''}
        ${canEdit ? '<button data-action="edit">Edit</button>' : ''}
        ${canDelete ? '<button data-action="delete">Delete</button>' : ''}
        ${hasAnchor ? '<button data-action="scroll">Go to</button>' : ''}
        <button data-action="logs">Show logs</button>
        <button data-action="comments">Show comments</button>
      </div>
      <div class="rn-logs" hidden></div>
      <div class="rn-comments" hidden>
        <div class="rn-comments-list"></div>
        <div class="rn-comments-add">
          <input type="text" placeholder="Add a comment..." class="rn-comment-input" />
          <button class="rn-comment-submit">Add</button>
        </div>
      </div>
    `;
    attachItemHandlers(div);
    return div;
  }

  function parseFilters(text) {
    return (text || '')
      .split(/[,;\s]+/)
      .map((t) => t.trim().toLowerCase())
      .filter(Boolean);
  }

  function matchesFilter(note, filterText) {
    const filters = parseFilters(filterText);
    if (filters.length === 0) return true;
    const tags = (note.tags || '')
      .split(',')
      .map((t) => t.trim().toLowerCase())
      .filter(Boolean);
    return filters.some((f) => tags.includes(f));
  }

  async function addComment(noteId, comment) {
    const base = (endpoints.commentsAdd || '').replace(
      /\/?0\/?comments\/?add$/,
      '',
    );
    const url = `${base}/${noteId}/comments/add`;
    return requestJSON(url, {
      method: 'POST',
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify({ comment }),
    });
  }

  async function addNote(payload) {
    return requestJSON(endpoints.add, {
      method: 'POST',
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify(payload),
    });
  }

  function enablePicker(onPicked) {
    if (isPicking) return;
    isPicking = true;
    document.body.classList.add('rn-picking');

    const onMove = (e) => {
      if (!isPicking) return;
      const t0 = normalizeTarget(e);
      const t = t0 && t0.closest('.rn-overlay') ? null : t0;
      if (t === currentHoverEl) return;
      if (currentHoverEl) currentHoverEl.classList.remove('rn-pick-highlight');
      currentHoverEl = t instanceof Element ? t : null;
      if (currentHoverEl) currentHoverEl.classList.add('rn-pick-highlight');
    };

    const onClick = async (e) => {
      if (!isPicking) return;
      const t0 = normalizeTarget(e);
      if (t0 && t0.closest('.rn-overlay')) return;
      e.preventDefault();
      e.stopPropagation();
      const el = t0 instanceof Element ? t0 : null;
      if (!el) return;
      const anchor = serializeAnchorFromElement(el);
      const selector = anchor ? anchor.selector : '';
      const note = prompt('Enter note text:');
      if (note == null) {
        disablePicker();
        return;
      }
      const trimmedNote = note.trim();
      if (trimmedNote === '') {
        alert('Oops! Looks like you forgot to add the note text.');
        disablePicker();
        return;
      }
      const tags = prompt('Enter tags (comma-separated):') || '';
      try {
        await addNote({
          path: currentPath,
          selector,
          anchor,
          note: trimmedNote,
          tags,
        });
      } catch (err) {
        alert(`Failed to add note: ${err.message}`);
      }
      disablePicker();
      const event = new CustomEvent('rn-refresh');
      window.dispatchEvent(event);
    };

    const onKey = (e) => {
      if (e.key === 'Escape') {
        disablePicker();
      }
    };

    document.addEventListener('mousemove', onMove, true);
    document.addEventListener('click', onClick, true);
    document.addEventListener('keydown', onKey, true);

    detachPickerHandlers = () => {
      document.removeEventListener('mousemove', onMove, true);
      document.removeEventListener('click', onClick, true);
      document.removeEventListener('keydown', onKey, true);
      detachPickerHandlers = null;
    };
  }

  async function boot() {
    // Get configuration
    const settings = drupalSettings.reviewerNotes || {};
    const highlightColor = settings.highlightAllColor || '#1e40af';
    const pickColor = settings.pickElementColor || '#c2410c';
    const overlayHeight = settings.overlayHeight || 35; // Default to 35 if not set.

    // Set CSS variables
    document.documentElement.style.setProperty(
      '--reviewer-notes-highlight-color',
      highlightColor,
    );
    document.documentElement.style.setProperty(
      '--reviewer-notes-pick-color',
      pickColor,
    );
    document.documentElement.style.setProperty(
      '--reviewer-notes-overlay-height',
      `${overlayHeight}vh`,
    );

    const overlay = createOverlay();
    const listEl = overlay.querySelector('.rn-list');
    const filterEl = overlay.querySelector('.rn-filter');
    const filterClearEl = overlay.querySelector('.rn-filter-clear');
    const countEl = overlay.querySelector('.rn-count');
    const countStatusEl = overlay.querySelector('.rn-count-status');
    const highlightBtn = overlay.querySelector('.rn-highlight-all');

    function updateClearVisibility() {
      if (filterClearEl) {
        filterClearEl.hidden = !(filterEl.value && filterEl.value.length > 0);
      }
    }

    async function refresh() {
      const notes = await getNotes();
      const filterText = filterEl.value;
      const filtered = notes.filter((n) => matchesFilter(n, filterText));

      listEl.innerHTML = '';
      filtered.forEach((note) => {
        const item = buildNoteItem(note);
        listEl.appendChild(item);
      });

      if (countEl) {
        countEl.textContent = `${filtered.length}/${notes.length} notes listed below`;
      }
      if (countStatusEl) {
        const openCount = filtered.filter(
          (n) => (n.status || '').toLowerCase() === 'open',
        ).length;
        const resolvedCount = filtered.filter(
          (n) => (n.status || '').toLowerCase() === 'resolved',
        ).length;
        countStatusEl.textContent = `open: ${openCount} · resolved: ${resolvedCount}`;
      }

      updateClearVisibility();

      // Show/hide highlight button based on whether there are notes with selectors.
      if (highlightBtn) {
        const hasAnchoredNotes = notes.some((n) =>
          (n.selector || (n.anchor && n.anchor.selector) || '').trim(),
        );
        highlightBtn.style.display = hasAnchoredNotes ? 'inline-block' : 'none';
        // If hiding button, also clear any active highlights.
        if (!hasAnchoredNotes && highlightAllActive) {
          highlightAllActive = false;
          clearAllHighlights();
        }
      }

      // Re-apply global highlights after each refresh if active.
      if (highlightAllActive) {
        await highlightAllAnnotations(notes);
      }
    }

    window.addEventListener('rn-refresh', refresh);

    filterEl.addEventListener('input', () => {
      updateClearVisibility();
      refresh();
    });
    if (filterClearEl) {
      filterClearEl.addEventListener('click', () => {
        if (filterEl.value) {
          filterEl.value = '';
          filterEl.dispatchEvent(new Event('input', { bubbles: true }));
        } else {
          updateClearVisibility();
        }
        filterEl.focus();
      });
      updateClearVisibility();
    }

    if (canAdd) {
      const pickBtn = overlay.querySelector('.rn-pick');
      pickBtn.addEventListener('click', () => {
        if (isPicking) {
          disablePicker();
        } else {
          enablePicker();
        }
      });

      overlay.querySelector('.rn-add').addEventListener('click', async (e) => {
        const note = prompt('Enter note text:');
        if (note == null) return;
        const trimmedNote = note.trim();
        if (trimmedNote === '') {
          alert('Oops! Looks like you forgot to add the note text.');
          return;
        }
        const tags = prompt('Enter tags (comma-separated):') || '';
        try {
          await addNote({
            path: currentPath,
            selector: '',
            anchor: null,
            note: trimmedNote,
            tags,
          });
          await refresh();
        } catch (err) {
          alert(`Failed to add note: ${err.message}`);
        }
      });
    }

    if (highlightBtn) {
      highlightBtn.addEventListener('click', async () => {
        if (!highlightAllActive) {
          highlightAllActive = true;
          highlightBtn.textContent = 'Clear highlights';
          await highlightAllAnnotations();
        } else {
          highlightAllActive = false;
          highlightBtn.textContent = 'Highlight all';
          clearAllHighlights();
        }
      });
    }

    // Keep delegated handler as a fallback.
    listEl.addEventListener('click', async (e) => {
      const base = normalizeTarget(e);
      if (!base) return;
      const btn = base.closest('button');
      // Fallback handler - specific button clicks are handled by other listeners
    });

    // Clicking a tag adds it to the filter and refreshes the list.
    listEl.addEventListener('click', (e) => {
      const base = normalizeTarget(e);
      if (!base) return;
      const tagEl = base.closest('.rn-tag');
      if (!tagEl) return;
      e.preventDefault();
      e.stopPropagation();
      const tag = (tagEl.textContent || '').trim();
      if (!tag) return;
      const existing = filterEl.value || '';
      const existingParts = parseFilters(existing);
      if (!existingParts.includes(tag.toLowerCase())) {
        const sep = existing.trim().length > 0 ? ', ' : '';
        filterEl.value = existing + sep + tag;
        filterEl.dispatchEvent(new Event('input', { bubbles: true }));
      } else {
        // Trigger refresh anyway to reflect current state.
        filterEl.dispatchEvent(new Event('input', { bubbles: true }));
      }
      filterEl.focus();
    });

    // Comment add handler (event delegation).
    overlay.addEventListener('click', async (e) => {
      const base = normalizeTarget(e);
      if (!base) return;
      const submit = base.closest('.rn-comment-submit');
      if (!submit) return;
      const wrap = base.closest('.rn-comments');
      if (!wrap) return;
      const parent = base.closest('.rn-item');
      const id = parent ? parseInt(parent.dataset.id, 10) : 0;
      const input = wrap.querySelector('.rn-comment-input');
      const value = ((input && input.value) || '').trim();
      if (!id || !value) return;
      try {
        await addComment(id, value);
        input.value = '';
        const { comments } = await listComments(id);
        const list = wrap.querySelector('.rn-comments-list');
        list.innerHTML = renderComments(comments);
        wrap.setAttribute('data-loaded', '1');
        const logsEl = parent.querySelector('.rn-logs');
        if (logsEl && logsEl.hasAttribute('data-loaded')) {
          const logs = await getLogs(id);
          logsEl.innerHTML = renderLogs(logs);
        }
      } catch (err) {
        alert(`Failed to add comment: ${err.message}`);
      }
    });

    await refresh();
  }

  Drupal.behaviors.reviewerNotes = {
    attach(context) {
      once('reviewer-notes', 'html', context).forEach(boot);
    },
  };
})(Drupal, drupalSettings, once);
