<?php

namespace Drupal\reviewer_notes\Form;

use Drupal\Core\Form\ConfirmFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;
use Drupal\Component\Utility\Html;
use Drupal\Core\Database\Connection;
use Symfony\Component\HttpFoundation\RequestStack;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a confirmation form for bulk deleting reviewer notes.
 *
 * This form displays a list of selected notes and asks for confirmation
 * before deleting them from the database.
 */
class ReviewerNotesBulkDeleteConfirmForm extends ConfirmFormBase {
  /**
   * The IDs of notes to be deleted.
   *
   * @var array
   */
  protected array $ids = [];

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected Connection $database;

  /**
   * The request stack.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  protected $requestStack;

  /**
   * Constructs a ReviewerNotesBulkDeleteConfirmForm object.
   *
   * @param \Drupal\Core\Database\Connection $database
   *   The database connection.
   * @param \Symfony\Component\HttpFoundation\RequestStack $request_stack
   *   The request stack.
   */
  public function __construct(Connection $database, RequestStack $request_stack) {
    $this->database = $database;
    $this->requestStack = $request_stack;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('database'),
      $container->get('request_stack')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'reviewer_notes_bulk_delete_confirm_form';
  }

  /**
   * {@inheritdoc}
   */
  public function getQuestion() {
    return $this->t('Are you sure you want to delete @count selected note(s)?', ['@count' => count($this->ids)]);
  }

  /**
   * {@inheritdoc}
   */
  public function getCancelUrl() {
    return Url::fromRoute('reviewer_notes.admin_manage');
  }

  /**
   * {@inheritdoc}
   */
  public function getConfirmText() {
    return $this->t('Delete selected');
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $request = $this->requestStack->getCurrentRequest();
    $ids_param = (string) $request->query->get('ids');
    $this->ids = array_values(array_filter(array_map('intval', explode(',', $ids_param)), static fn($v) => $v > 0));

    if (!empty($this->ids)) {
      $rows = $this->database->select('reviewer_notes', 'rn')
        ->fields('rn', ['id', 'path', 'note', 'tags', 'status'])
        ->condition('id', $this->ids, 'IN')
        ->orderBy('id', 'DESC')
        ->execute()
        ->fetchAll();
      $items = [];
      foreach ($rows as $r) {
        $items[] = '#' . (int) $r->id . ' — ' . Html::escape($r->status) . ' · ' . Html::escape($r->path) . ' · ' . Html::escape($r->tags) . ' — "' . Html::escape($r->note) . '"';
      }
      if ($items) {
        $form['preview'] = [
          '#type' => 'details',
          '#title' => $this->t('Selected notes'),
          '#open' => TRUE,
        ];
        $form['preview']['list'] = [
          '#theme' => 'item_list',
          '#items' => $items,
        ];
      }
    }

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    if (!empty($this->ids)) {
      $this->database->delete('reviewer_notes')->condition('id', $this->ids, 'IN')->execute();
      $this->messenger()->addStatus($this->t('Deleted @count note(s).', ['@count' => count($this->ids)]));
    }
    $form_state->setRedirectUrl($this->getCancelUrl());
  }

}
