<?php

namespace Drupal\reviewer_notes\Form;

use Drupal\Core\Form\ConfirmFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;
use Drupal\Component\Utility\Html;
use Drupal\Core\Database\Connection;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a confirmation form for deleting a single reviewer note.
 *
 * This form displays the details of the note to be deleted and asks for
 * confirmation before proceeding with the delete operation.
 */
class ReviewerNotesDeleteConfirmForm extends ConfirmFormBase {
  /**
   * The ID of the note to be deleted.
   *
   * @var int
   */
  protected int $id;

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected Connection $database;

  /**
   * Constructs a ReviewerNotesDeleteConfirmForm object.
   *
   * @param \Drupal\Core\Database\Connection $database
   *   The database connection.
   */
  public function __construct(Connection $database) {
    $this->database = $database;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('database')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'reviewer_notes_delete_confirm_form';
  }

  /**
   * {@inheritdoc}
   */
  public function getQuestion() {
    return $this->t('Are you sure you want to delete note #%id?', ['%id' => $this->id]);
  }

  /**
   * {@inheritdoc}
   */
  public function getCancelUrl() {
    return Url::fromRoute('reviewer_notes.admin_manage');
  }

  /**
   * {@inheritdoc}
   */
  public function getConfirmText() {
    return $this->t('Delete');
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, $id = NULL) {
    $this->id = (int) $id;

    // Load note preview details.
    $note = NULL;
    if ($this->id > 0) {
      $note = $this->database->select('reviewer_notes', 'rn')
        ->fields('rn', ['id', 'path', 'note', 'tags', 'status'])
        ->condition('id', $this->id)
        ->execute()
        ->fetchObject();
    }

    if ($note) {
      $line = '#' . (int) $note->id . ' — ' . Html::escape($note->status)
        . ' · ' . Html::escape($note->path)
        . (!empty($note->tags) ? ' · ' . Html::escape($note->tags) : '')
        . ' — "' . Html::escape($note->note) . '"';

      $form['preview'] = [
        '#type' => 'details',
        '#title' => $this->t('Selected note'),
        '#open' => TRUE,
      ];
      $form['preview']['list'] = [
        '#theme' => 'item_list',
        '#items' => [$line],
      ];
    }

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    if ($this->id > 0) {
      $this->database->delete('reviewer_notes')->condition('id', $this->id)->execute();
      $this->messenger()->addStatus($this->t('Deleted note #%id.', ['%id' => $this->id]));
    }
    $form_state->setRedirectUrl($this->getCancelUrl());
  }

}
