<?php

namespace Drupal\reviewer_notes\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;

/**
 * Provides a configuration form for reviewer notes settings.
 *
 * This form allows administrators to configure various settings for the
 * reviewer notes module, including URL patterns and overlay height.
 */
class ReviewerNotesSettingsForm extends ConfigFormBase {

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['reviewer_notes.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'reviewer_notes_settings_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('reviewer_notes.settings');

    $form['enabled'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable Reviewer Notes'),
      '#default_value' => (bool) $config->get('enabled'),
      '#description' => $this->t('Master switch. When disabled, the overlay never appears even if URL rules match.'),
    ];

    $form['csrf_protect'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Require CSRF token for POST endpoints'),
      '#default_value' => (bool) $config->get('csrf_protect', TRUE),
      '#description' => $this->t('Strongly recommended for production. Disable only for development/testing if you understand the risks.'),
    ];

    $form['overlay_height'] = [
      '#type' => 'number',
      '#title' => $this->t('Overlay Height (in vh)'),
      '#min' => 10,
      '#max' => 90,
      '#default_value' => $config->get('overlay_height') ?: 35,
      '#description' => $this->t('The maximum height of the reviewer notes overlay, in viewport height units (vh).'),
    ];

    $form['urls'] = [
      '#type' => 'details',
      '#title' => $this->t('URL rules'),
      '#open' => TRUE,
      '#description' => $this->t('Rules are evaluated against both the current URL alias (e.g. /about-us) and the internal system path (e.g. /node/123). If no patterns are provided, the overlay is allowed everywhere (subject to permissions).'),
    ];

    $form['urls']['url_patterns'] = [
      '#type' => 'textarea',
      '#title' => $this->t('URL patterns'),
      '#description' => $this->t('Enter one pattern per line. The * wildcard matches any characters across segments. Examples: @ex1, @ex2. Patterns are checked against both the alias and the system path.', [
        '@ex1' => '/node/*',
        '@ex2' => '/about*',
      ]),
      '#default_value' => implode("\n", (array) $config->get('url_patterns') ?: []),
    ];

    $form['urls']['negate_urls'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Negate these URL patterns'),
      '#description' => $this->t('When checked, the list works as an exclusion: the overlay is shown everywhere except on matching URLs.'),
      '#default_value' => (bool) $config->get('negate_urls'),
    ];

    $form['colors'] = [
      '#type' => 'details',
      '#title' => $this->t('Highlight Colors'),
      '#open' => TRUE,
      '#description' => $this->t('Configure colors used for highlighting elements.'),
    ];

    $form['colors']['highlight_all_color'] = [
      '#type' => 'color',
      '#title' => $this->t('Highlight All Color'),
      '#default_value' => $config->get('highlight_all_color') ?: '#1e40af',
      '#description' => $this->t('Color used when highlighting all notes.'),
    ];

    $form['colors']['pick_element_color'] = [
      '#type' => 'color',
      '#title' => $this->t('Pick Element Color'),
      '#default_value' => $config->get('pick_element_color') ?: '#c2410c',
      '#description' => $this->t('Color used when picking elements.'),
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    parent::submitForm($form, $form_state);
    $patterns_text = $form_state->getValue('url_patterns') ?: '';
    $patterns = array_values(array_filter(array_map('trim', preg_split('/\r?\n/', $patterns_text)), static function ($v) {
      return $v !== '';
    }));

    $this->configFactory()->getEditable('reviewer_notes.settings')
      ->set('enabled', (bool) $form_state->getValue('enabled'))
      ->set('csrf_protect', (bool) $form_state->getValue('csrf_protect'))
      ->set('url_patterns', $patterns)
      ->set('negate_urls', (bool) $form_state->getValue('negate_urls'))
      ->set('highlight_all_color', $form_state->getValue('highlight_all_color'))
      ->set('pick_element_color', $form_state->getValue('pick_element_color'))
      ->set('overlay_height', (int) $form_state->getValue('overlay_height'))
      ->clear('enabled_content_types')
      ->save();

    $this->messenger()->addStatus($this->t('Please clear caches on the <a href=":url">Performance</a> page.', [
      ':url' => Url::fromRoute('system.performance_settings')->toString(),
    ]));
  }

}
