<?php

namespace Drupal\reviewer_notes;

use Drupal\Core\Database\Connection;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\Component\Utility\Xss;
use Drupal\Component\Datetime\TimeInterface;

/**
 * Service for managing reviewer notes.
 */
class ReviewerNotesService {

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface
   */
  protected $currentUser;

  /**
   * The time service.
   *
   * @var \Drupal\Component\Datetime\TimeInterface
   */
  protected TimeInterface $time;

  /**
   * Constructs a new ReviewerNotesService.
   *
   * @param \Drupal\Core\Database\Connection $database
   *   The database connection.
   * @param \Drupal\Core\Session\AccountProxyInterface $current_user
   *   The current user.
   * @param \Drupal\Component\Datetime\TimeInterface $time
   *   The time service.
   */
  public function __construct(Connection $database, AccountProxyInterface $current_user, TimeInterface $time) {
    $this->database = $database;
    $this->currentUser = $current_user;
    $this->time = $time;
  }

  /**
   * Retrieves a single reviewer note by its ID.
   *
   * @param int $id
   *   The ID of the note to retrieve.
   *
   * @return array|null
   *   An associative array representing the note, or NULL if not found.
   */
  public function getNoteById(int $id): ?array {
    $query = $this->database->select('reviewer_notes', 'rn')
      ->fields('rn')
      ->condition('id', $id);
    $result = $query->execute()->fetchAssoc();

    if ($result) {
      $result['id'] = (int) $result['id'];
      $result['uid'] = (int) $result['uid'];
      $result['created'] = (int) $result['created'];
      $result['changed'] = (int) $result['changed'];
      $result['anchor'] = $result['anchor_json'] ? json_decode($result['anchor_json'], TRUE) : NULL;
      unset($result['anchor_json']);
    }

    return $result ?: NULL;
  }

  /**
   * Adds a new reviewer note.
   *
   * @param array $payload
   *   An associative array containing note data.
   *
   * @return int
   *   The ID of the newly created note.
   *
   * @throws \Exception
   *   If path or note are missing from the payload.
   */
  public function addNote(array $payload): int {
    $path = (string) ($payload['path'] ?? '');
    $selector = (string) ($payload['selector'] ?? '');
    $anchor = $payload['anchor'] ?? NULL;
    $note = Xss::filter((string) ($payload['note'] ?? ''));
    $rawTags = (string) ($payload['tags'] ?? '');

    if ($path === '' || $note === '') {
      throw new \Exception('Path and note are required.');
    }

    $tags = $this->normalizeTags($rawTags);

    $time = $this->time->getRequestTime();
    $id = $this->database->insert('reviewer_notes')->fields([
      'uid' => (int) $this->currentUser->id(),
      'path' => $path,
      'selector' => $selector,
      'anchor_json' => $anchor ? json_encode($anchor) : NULL,
      'note' => $note,
      'tags' => $tags,
      'status' => 'open',
      'created' => $time,
      'changed' => $time,
    ])->execute();

    $this->logActivity((int) $id, 'create', [
      'note_id' => (int) $id,
      'path' => $path,
      'note' => $note,
      'tags' => $tags,
    ]);

    return (int) $id;
  }

  /**
   * Updates an existing reviewer note.
   *
   * @param array $payload
   *   An associative array containing note data to update.
   *
   * @return bool
   *   TRUE if the note was updated, FALSE otherwise.
   *
   * @throws \Exception
   *   If id is missing or invalid.
   */
  public function updateNote(array $payload): bool {
    $id = (int) ($payload['id'] ?? 0);
    if ($id <= 0) {
      throw new \Exception('Invalid id');
    }

    $fields = [];
    $changes = [];
    if (isset($payload['note'])) {
      $fields['note'] = Xss::filter((string) $payload['note']);
      $changes['note'] = $fields['note'];
    }
    if (isset($payload['tags'])) {
      $fields['tags'] = $this->normalizeTags((string) $payload['tags']);
      $changes['tags'] = $fields['tags'];
    }
    if (isset($payload['status'])) {
      $status = (string) $payload['status'];
      if (!in_array($status, ['open', 'resolved'], TRUE)) {
        throw new \Exception('Invalid status');
      }
      $fields['status'] = $status;
      $changes['status'] = $status;
    }

    if (empty($fields)) {
      throw new \Exception('No fields to update');
    }

    $fields['changed'] = $this->time->getRequestTime();

    $updated = (bool) $this->database->update('reviewer_notes')
      ->fields($fields)
      ->condition('id', $id)
      ->execute();

    if ($updated) {
      $this->logActivity($id, 'update', $changes);
    }

    return $updated;
  }

  /**
   * Deletes a reviewer note by its ID.
   *
   * @param int $id
   *   The ID of the note to delete.
   *
   * @return bool
   *   TRUE if the note was deleted, FALSE otherwise.
   *
   * @throws \Exception
   *   If id is invalid.
   */
  public function deleteNote(int $id): bool {
    if ($id <= 0) {
      throw new \Exception('Invalid id for deletion.');
    }

    // First, delete all related comments.
    $this->database->delete('reviewer_notes_comment')
      ->condition('note_id', $id)
      ->execute();

    // Then, delete all related log entries.
    $this->database->delete('reviewer_notes_log')
      ->condition('note_id', $id)
      ->execute();

    // Finally, delete the note itself.
    $deleted = (bool) $this->database->delete('reviewer_notes')
      ->condition('id', $id)
      ->execute();

    // Note: We don't log the deletion activity here since we're deleting
    // the log entries above. The frontend JavaScript already handles
    // the user feedback for successful deletion.
    return $deleted;
  }

  /**
   * Normalizes a comma-separated string of tags into a consistent format.
   *
   * @param string $raw
   *   The raw tags string.
   *
   * @return string
   *   The normalized tags string.
   */
  private function normalizeTags(string $raw): string {
    $parts = preg_split('/[\s,;]+/u', $raw) ?: [];
    $clean = [];
    foreach ($parts as $p) {
      $t = trim($p);
      if ($t === '') {
        continue;
      }
      if (!in_array($t, $clean, TRUE)) {
        $clean[] = $t;
      }
    }
    return implode(', ', $clean);
  }

  /**
   * Logs activity for a reviewer note.
   *
   * @param int $noteId
   *   The ID of the note.
   * @param string $action
   *   The action performed (e.g., 'create', 'update', 'delete', 'comment').
   * @param array $details
   *   Additional details about the activity.
   */
  public function logActivity(int $noteId, string $action, array $details): void {
    $time = $this->time->getRequestTime();
    $this->database->insert('reviewer_notes_log')->fields([
      'note_id' => $noteId,
      'uid' => (int) $this->currentUser->id(),
      'action' => $action,
      'details' => $details ? json_encode($details) : NULL,
      'created' => $time,
    ])->execute();
  }

}
