<?php

namespace Drupal\revision_extras\Form;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\TypedConfigManagerInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandler;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\ConfigTarget;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Form\RedundantEditableConfigNamesTrait;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Form for configuring Revision Extras settings.
 *
 * @package Drupal\revision_extras\Form
 */
class RevisionExtrasSettingsForm extends ConfigFormBase {
  use RedundantEditableConfigNamesTrait;

  /**
   * The entityTypeManager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The module handler service.
   *
   * @var \Drupal\Core\Extension\ModuleHandler
   */
  protected ModuleHandler $moduleHandler;

  /**
   * Constructs a Revisions Extras settings form object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory service.
   * @param \Drupal\Core\Config\TypedConfigManagerInterface $typedConfigManager
   *   The typed config manager.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entityTypeManager.
   * @param \Drupal\Core\Extension\ModuleHandler $moduleHandler
   *   Module handler service.
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    TypedConfigManagerInterface $typedConfigManager,
    EntityTypeManagerInterface $entityTypeManager,
    ModuleHandler $moduleHandler,
  ) {
    parent::__construct($config_factory, $typedConfigManager);
    $this->entityTypeManager = $entityTypeManager;
    $this->moduleHandler = $moduleHandler;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory'),
      $container->get('config.typed'),
      $container->get('entity_type.manager'),
      $container->get('module_handler'),
    );
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return [
      'revision_extras.settings',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'revision_extras_settings';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $form['form_text_changes'] = [
      '#type' => 'details',
      '#title' => $this->t('Revision Log Message Form Appearance'),
      '#description' => $this->t('Customizations apply in all form instances where the revision log message field appears. Leave blank to use the default label and description.'),
      '#description_display' => 'before',
      '#open' => TRUE,
    ];
    $form['form_text_changes']['revision_log_message_label'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Label for the revision log message field'),
      '#config_target' => new ConfigTarget(
        'revision_extras.settings',
        'revision_log_message_label',
      ),
      '#maxlength' => 255,
      '#description' => $this->t('Leave blank to use the default label.'),
    ];
    $form['form_text_changes']['revision_log_message_description'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Description text for the revision log message field'),
      '#config_target' => new ConfigTarget(
        'revision_extras.settings',
        'revision_log_message_description',
      ),
      '#maxlength' => 255,
      '#description' => $this->t('Leave blank to use the default description.'),
    ];
    $form['form_text_changes']['append_last_log_message'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Append the last revision log message to the field description if it exists.'),
      '#config_target' => new ConfigTarget(
        'revision_extras.settings',
        'append_last_log_message',
      ),
      '#description' => $this->t('Display the last revision log message at the end of the revision log message field description.'),
    ];
    $form['form_text_changes']['last_log_message_label'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Label for the last revision log message in the description'),
      '#config_target' => new ConfigTarget(
        'revision_extras.settings',
        'last_log_message_label',
      ),
      '#maxlength' => 255,
      '#required' => TRUE,
      '#description' => $this->t('This only applies if appending the last revision log message to the field description.'),
      '#states' => [
        'visible' => [
          ':input[name="append_last_log_message"]' => ['checked' => TRUE],
        ],
      ],
    ];

    // Allow site builders to select which entity types require revisions and
    // log messages.
    $entity_types = [
      'node' => 'Content',
      'media' => 'Media',
      'block_content' => 'Block',
      // Consider adding more revisionable types.
    ];
    foreach ($entity_types as $entity_type => $entity_label) {
      // Make sure the corresponding core module is enabled before showing the
      // entity as a configuration option.
      if ($this->moduleHandler->moduleExists($entity_type)) {
        $typeStorage = $this->entityTypeManager->getStorage($entity_type . '_type');
        $entity_definition = $typeStorage->getEntityType();
        $plural_type_label = $entity_definition->getPluralLabel()->getUntranslatedString();
        $bundles = $typeStorage->loadMultiple() ?? [];

        // Create an array of bundle options for this type.
        $bundle_options = [];
        foreach ($bundles as $bundle) {
          $bundle_options[$bundle->id()] = $bundle->label();
        }
        if (!empty($bundle_options)) {
          $form['require_' . $entity_type . '_wrapper'] = [
            '#type' => 'details',
            '#title' => $this->t('@label Requirements', ['@label' => $entity_label]),
            '#open' => TRUE,
          ];
          $form['require_' . $entity_type . '_wrapper'][$entity_type . '_bundles'] = [
            '#type' => 'checkboxes',
            '#title' => $this->t('@label', ['@label' => ucfirst($plural_type_label)]),
            '#description' => $this->t('Select the @label that require new revisions and revision log messages. If none are selected, then they will not be required.', ['@label' => $plural_type_label]),
            // Checkboxes don't accept a null default value.
            '#default_value' => $this->config('revision_extras.settings')->get($entity_type . '_bundles') ?? [],
            '#config_target' => new ConfigTarget(
              'revision_extras.settings',
              $entity_type . '_bundles',
            ),
            '#options' => $bundle_options,
          ];
          $form['require_' . $entity_type . '_wrapper']['enable_on_new_' . $entity_type] = [
            '#type' => 'checkbox',
            '#title' => $this->t('Require the revision log message on the "add" forms.'),
            '#config_target' => new ConfigTarget(
              'revision_extras.settings',
              'enable_on_new_' . $entity_type,
            ),
            '#description' => $this->t('Applies to all the above @label that require revision log messages.', ['@label' => $plural_type_label]),
          ];
          // Add option to populate default message in media library add form.
          if ($entity_type === 'media' && $this->moduleHandler->moduleExists('media_library')) {
            $form['require_' . $entity_type . '_wrapper']['populate_default_in_media_library'] = [
              '#type' => 'radios',
              '#title' => $this->t('Choose a method for populating a default log message when adding via the media library form. (Experimental)'),
              '#options' => [
                'none' => $this->t('Do not populate a default log message.'),
                'static' => $this->t('Static: "Added via the media library form."'),
                'parent_reference' => $this->t('Reference Parent (with fallback): "Added via the media library form in "[parent title]" (ID: [parent ID])."'),
                'parent_last_message' => $this->t('Previous revision log message of the parent entity.'),
              ],
              '#config_target' => new ConfigTarget(
                'revision_extras.settings',
                'populate_default_in_media_library',
              ),
              '#description' => $this->t('Only applies if the log message is required when adding media. Note that the parent context is not available when adding via a WYSIWYG and only exists if the parent has been saved. For media in layout builder blocks, the parent is the entity with the layout.'),
              '#states' => [
                'visible' => [
                  ':input[name="enable_on_new_media"]' => ['checked' => TRUE],
                ],
              ],
            ];
          }
        }
      }
    }

    return parent::buildForm($form, $form_state);
  }

}
