<?php

declare(strict_types=1);

namespace Drupal\Tests\revision_extras\FunctionalJavascript;

use Drupal\media\Entity\Media;
use Drupal\Tests\media_library\FunctionalJavascript\MediaLibraryTestBase;
use Drupal\Tests\TestFileCreationTrait;

/**
 * Test that uploads in the 'media_library_widget' work as expected.
 *
 * Based on media_library WidgetUploadTest.
 *
 * @group revision_extras
 */
class MediaLibraryRevisionTest extends MediaLibraryTestBase {

  use TestFileCreationTrait;

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * {@inheritdoc}
   */
  protected static $modules = ['media_library_test', 'hold_test', 'revision_extras'];

  /**
   * The test file.
   *
   * @var object
   */
  protected $testFile;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Get a test file.
    foreach ($this->getTestFiles('image') as $image) {
      $extension = pathinfo($image->filename, PATHINFO_EXTENSION);
      if ($extension === 'png') {
        $this->testFile = $image;
        continue;
      }
    }

    if (!isset($this->testFile)) {
      $this->fail('Expected test file not present.');
    }

    // Create a user that can create media and view revisions.
    $user = $this->drupalCreateUser([
      'administer site configuration',
      'access administration pages',
      'access content',
      'create basic_page content',
      'create media',
      'view media',
      'view all media revisions',
    ], 'media_user');
    $this->drupalLogin($user);
  }

  /**
   * Test revision user is set after creating media via media library widget.
   */
  public function testMediaLibraryRevisionUser(): void {
    $assert_session = $this->assertSession();
    $page = $this->getSession()->getPage();

    // Visit a node create page and open the media library.
    $this->drupalGet('node/add/basic_page');

    /** @var \Drupal\Core\File\FileSystemInterface $file_system */
    $file_system = $this->container->get('file_system');

    // Add to the twin media field.
    $this->openMediaLibraryForField('field_twin_media');

    // Add media using the media library widget.
    $assert_session->fieldExists('Add files');
    $this->addMediaFileToField('Add files', $file_system->realpath($this->testFile->uri));
    $this->assertMediaAdded();

    // Assert the revision_log_message field is not shown.
    $upload_form = $assert_session->elementExists('css', '.js-media-library-add-form');
    $assert_session->fieldNotExists('Revision log message', $upload_form);
    $page->fillField('Alternative text', $this->randomString());
    $this->pressSaveButton();

    // Load the created media item.
    $media_items = Media::loadMultiple();
    $added_media = array_pop($media_items);

    // View the revisions of the added media.
    $this->drupalGet('media/' . $added_media->id() . '/revisions');

    // The user's name should be in the revision log.
    $assert_session->pageTextContains('media_user');

    // The user's name should not be anonymous.
    $assert_session->pageTextNotContains('Anonymous');
  }

  /**
   * Test that the log message text is customized.
   */
  public function testLogMessageFieldTextCustomized(): void {
    $assert_session = $this->assertSession();

    // Navigate to the settings form.
    $this->drupalGet('/admin/config/user-interface/revision-extras');
    $page = $this->getSession()->getPage();

    // Customize the field label and description.
    $custom_label = 'Unique custom label';
    $custom_description = 'Unique custom description';
    $page->fillField('revision_log_message_label', $custom_label);
    $page->fillField('revision_log_message_description', $custom_description);

    // Require log message on new media of type_three so field appears.
    $page->checkField('media_bundles[type_three]');
    $page->checkField('enable_on_new_media');
    $page->findButton('Save configuration')->click();
    $assert_session->pageTextContains('The configuration options have been saved.');

    // Visit a node create page and open the media library.
    $this->drupalGet('node/add/basic_page');

    /** @var \Drupal\Core\File\FileSystemInterface $file_system */
    $file_system = $this->container->get('file_system');

    // Add to the twin media field.
    $this->openMediaLibraryForField('field_twin_media');
    $this->switchToMediaType('Three');

    // Add media using the media library widget.
    $choose_files = $assert_session->elementExists('css', '.form-managed-file');
    $choose_files->hasButton('Choose files');
    $this->addMediaFileToField('Add files', $file_system->realpath($this->testFile->uri));
    $this->assertMediaAdded();
    $upload_form = $assert_session->elementExists('css', '.js-media-library-add-form');
    $page->fillField('Alternative text', $this->randomString());

    // Assert the revision_log_message field IS shown with custom label.
    $assert_session->fieldExists($custom_label, $upload_form);

    // Assert the custom description is present.
    $assert_session->pageTextContains($custom_description);

    // Assert that the form errors with empty log message.
    $this->pressSaveButton(TRUE);
    $this->waitForText($custom_label . ' field is required.');

    // Add a revision log message and save successfully.
    $page->fillField('media[0][fields][revision_log_message][0][value]', $this->randomString());
    $this->pressSaveButton();
    $this->assertJsCondition('jQuery("input[name=\'media_library_select_form[1]\']").is(":focus")');
    $this->pressInsertSelected('Added one media item.');

    // Change setting to require on media type_one, not type_three.
    $this->drupalGet('/admin/config/user-interface/revision-extras');
    $page->checkField('media_bundles[type_one]');
    $page->uncheckField('media_bundles[type_three]');
    $page->findButton('Save configuration')->click();
    $assert_session->pageTextContains('The configuration options have been saved.');

    // Visit a node create page and open the media library.
    $this->drupalGet('node/add/basic_page');

    $this->openMediaLibraryForField('field_twin_media');
    $this->switchToMediaType('Three');

    // Add media using the media library widget.
    $choose_files = $assert_session->elementExists('css', '.form-managed-file');
    $choose_files->hasButton('Choose files');
    $this->addMediaFileToField('Add files', $file_system->realpath($this->testFile->uri));
    $this->assertMediaAdded();

    $upload_form = $assert_session->elementExists('css', '.js-media-library-add-form');
    $page->fillField('Alternative text', $this->randomString());

    // Assert the revision_log_message field is NOT present.
    $assert_session->fieldNotExists($custom_label, $upload_form);
    $this->pressSaveButton();
    // Added media successfully without log message.
    $this->waitForText('Add or select media');
  }

}
