<?php

namespace Drupal\revision_extras;

use Drupal\Component\Plugin\Exception\ContextException;
use Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException;
use Drupal\Component\Plugin\Exception\PluginNotFoundException;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\ImmutableConfig;
use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\Entity\ContentEntityForm;
use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Entity\RevisionableInterface;
use Drupal\Core\Entity\RevisionLogInterface;
use Drupal\Core\Extension\ModuleHandler;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Routing\RouteMatchInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Class to alter revision log message in various forms.
 */
class RevisionExtrasFormAlter implements ContainerInjectionInterface {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The configuration factory service.
   *
   * @var \Drupal\Core\Config\ImmutableConfig
   */
  protected ImmutableConfig $config;

  /**
   * The route match service.
   *
   * @var \Drupal\Core\Routing\RouteMatchInterface
   */
  protected RouteMatchInterface $routeMatch;

  /**
   * The module handler service.
   *
   * @var \Drupal\Core\Extension\ModuleHandler
   */
  protected ModuleHandler $moduleHandler;

  /**
   * RevisionExtrasFormAlter constructor.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   Entity type manager.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The configuration factory service.
   * @param \Drupal\Core\Routing\RouteMatchInterface $routeMatch
   *   Route match service.
   * @param \Drupal\Core\Extension\ModuleHandler $moduleHandler
   *   Module handler service.
   */
  public function __construct(
    EntityTypeManagerInterface $entityTypeManager,
    ConfigFactoryInterface $config_factory,
    RouteMatchInterface $routeMatch,
    ModuleHandler $moduleHandler,
  ) {
    $this->entityTypeManager = $entityTypeManager;
    $this->config = $config_factory->get('revision_extras.settings');
    $this->routeMatch = $routeMatch;
    $this->moduleHandler = $moduleHandler;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('entity_type.manager'),
      $container->get('config.factory'),
      $container->get('current_route_match'),
      $container->get('module_handler'),
    );
  }

  /**
   * Alter revision log message field in content entity forms.
   *
   * @param array $form
   *   An associative array containing the structure of the form.
   * @param \Drupal\Core\Entity\ContentEntityForm $form_object
   *   The form object.
   */
  public function alterContentEntityForm(array &$form, ContentEntityForm $form_object): void {
    $entity = $form_object->getEntity();
    $entity_type = $entity->getEntityType();
    $entity_type_id = $entity->getEntityTypeId();
    if ($entity_type->showRevisionUi()) {
      // Get the name of the revision log message field for this entity type.
      $revision_log_message_form_item = $entity_type->getRevisionMetadataKey('revision_log_message');

      // Customize the field label if configured.
      $custom_label = $this->config->get('revision_log_message_label') ?? '';
      if (!empty($custom_label)) {
        $form[$revision_log_message_form_item]['widget'][0]['value']['#title'] = $custom_label;
      }

      // Customize the field description if configured.
      $custom_description = $this->config->get('revision_log_message_description') ?? '';
      if (!empty($custom_description)) {
        $form[$revision_log_message_form_item]['widget'][0]['value']['#description'] = $custom_description;
      }

      // Append last revision log message to field description if configured.
      $append_last = $this->config->get('append_last_log_message') ?? FALSE;
      if ($append_last && !$entity->isNew()) {
        $last_revision_log = $this->getLastRevisionLog($entity->id(), $entity_type_id);
        if ($last_revision_log) {
          $lastLabel = $this->config->get('last_log_message_label') ?: 'Last revision log message';
          $full_description = '<p>' . $form[$revision_log_message_form_item]['widget'][0]['value']['#description'] . '</p>';
          $full_description .= '<p>' . $lastLabel . ': <br />' . $last_revision_log . '</p>';
          $form[$revision_log_message_form_item]['widget'][0]['value']['#description'] = $full_description;
        }
      }

      // Make revisions and revision log message required as configured.
      $supported_types = ['node', 'block_content', 'media'];
      if (in_array($entity_type_id, $supported_types)) {
        $bundles = $this->config->get($entity_type_id . '_bundles') ?? [];

        if (in_array($entity->bundle(), $bundles)) {
          // Check if revision log message is required on new content.
          $enable_on_new = $this->config->get('enable_on_new_' . $entity_type_id) ?? FALSE;
          if (!$entity->isNew() || $enable_on_new) {
            // Make revisions required.
            $form['revision']['#default_value'] = TRUE;

            // Hide the revision checkbox.
            $form['revision']['#access'] = FALSE;

            // Require the revision log message.
            $form[$revision_log_message_form_item]['widget'][0]['value']['#required'] = TRUE;
          }
        }
      }
    }
  }

  /**
   * Alter revision log field the content moderation form.
   *
   * @param array $form
   *   An associative array containing the structure of the form.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   */
  public function alterContentModerationForm(array &$form, FormStateInterface $form_state): void {
    if (isset($form['revision_log'])) {
      // Customize the field label if configured.
      $custom_label = $this->config->get('revision_log_message_label') ?? '';
      if (!empty($custom_label)) {
        $form['revision_log']['#title'] = $custom_label;
      }

      // Get the entity from the form state.
      $storage = $form_state->getStorage();

      if (isset($storage['entity']) && $storage['entity'] instanceof RevisionLogInterface) {
        $entity = $storage['entity'];
        $entity_type_id = $entity->getEntityTypeId();
        // Make the log message field required if configured.
        $supported_types = ['node', 'block_content', 'media'];
        if (in_array($entity_type_id, $supported_types)) {
          $bundles = $this->config->get($entity_type_id . '_bundles') ?? [];

          if (in_array($entity->bundle(), $bundles)) {
            $form['revision_log']['#required'] = TRUE;
          }

          // Pre-fill with the last revision log message if available.
          $last_log_message = $entity->getRevisionLogMessage() ?? '';
          $form['revision_log']['#default_value'] = $last_log_message;
        }
      }
    }
  }

  /**
   * Alter revision log message field in the media library add form.
   *
   * @param array $form
   *   An associative array containing the structure of the form.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   *
   * @throws \Drupal\Component\Plugin\Exception\ContextException
   */
  public function alterMediaLibraryAddForm(array &$form, FormStateInterface $form_state): void {
    // By default, the revision log message does not appear in the media
    // library form.
    // So this only applies if the log message is required on new media.
    $bundles = $this->config->get('media_bundles') ?? [];
    $enable_on_new = $this->config->get('enable_on_new_media') ?? FALSE;

    if (!empty($bundles) && $enable_on_new) {
      // Get the selected media type and check if it requires log message.
      $storage = $form_state->getStorage();
      $is_wysiwyg = FALSE;
      $parent_entity_type_id = '';
      $parent_entity_id = '';
      // Avoid a dependency on media_library.
      // @phpcs:disable
      // @phpstan-ignore-next-line
      if (isset($storage['media_library_state']) &&
        $storage['media_library_state'] instanceof \Drupal\media_library\MediaLibraryState
      ) {
        $state = $storage['media_library_state'];
        $bundle = $state->getSelectedTypeId();
        // If this media type does not require revision log message, abort.
        if (!in_array($bundle, $bundles)) {
          return;
        }

        // Check if this was opened via the WYSIWYG editor.
        $opener = $state->getOpenerId();
        if ($opener === 'media_library.opener.editor') {
          $is_wysiwyg = TRUE;
        }
        else {
          // Try to get the parent entity from the opener parameters.
          // This only works for parent entities that are not new and already have an ID.
          $params = $state->getOpenerParameters();
          if (isset($params['entity_type_id']) && isset($params['entity_id'])) {
            $parent_entity_type_id = $params['entity_type_id'];
            $parent_entity_id = $params['entity_id'];
          }
        }
      }

      // Store a default log message from the parent context if configured to populate a default.
      $populate_default_in_media_library = $this->config->get('populate_default_in_media_library') ?? 'none';

      if ($populate_default_in_media_library !== 'none' &&
        !isset($form['revision_extras_default_log_message'])
      ) {
        // Set a static default log message.
        if ($populate_default_in_media_library === 'static' ||
          $populate_default_in_media_library === 'parent_reference'
        ) {
          $default_message = 'Added via the media library form.';
        }

        // For dynamic default log messages, try to get the parent entity.
        // The parent is not available from the WYSIWYG editor.
        if (!$is_wysiwyg && $populate_default_in_media_library !== 'static') {
          $parent = FALSE;
          // Use the media library opener parameters to get the parent entity.
          if (!empty($parent_entity_type_id) && !empty($parent_entity_id)) {
            $parent = $this->entityTypeManager->getStorage($parent_entity_type_id)->load($parent_entity_id);
          }
          // For media in layout builder blocks, the parent entity is available
          // from context.
          elseif($this->moduleHandler->moduleExists('layout_builder')) {
            try {
              $parent = $this->getMediaParentFromLayoutContext();
            }
            catch (ContextException $e) {
            }
          }

          if ($parent) {
            switch ($populate_default_in_media_library) {
              case 'parent_reference':
                $default_message = 'Added via the media library form in "' . $parent->label() . '" (ID: ' . $parent->id() . ').';
                break;

              case 'parent_last_message':
                if ($parent instanceof RevisionableInterface) {
                  $default_message = $this->getLastRevisionLog($parent->id());
                }
                break;
            }
          }
        }

        // Store the default revision log message in a hidden field.
        if (isset($default_message)) {
          $form['revision_extras_default_log_message'] = [
            '#type' => 'hidden',
            '#value' => $default_message,
          ];
        }
      }

      // Add a process function to make revision log message required.
      $form['#process'][] = '_revision_extras_media_library_log_message_required';
    }
  }

  /**
   * Get the last revision log message.
   *
   * @param int $id
   *   The entity ID.
   * @param string $entity_type
   *   The entity type.
   *
   * @return \Drupal\Core\Entity\RevisionableInterface|null
   *   The last entity revision, or NULL if not set.
   */
  protected function getLastRevision(int $id, string $entity_type): ?RevisionableInterface {
    try {
      /** @var \Drupal\Core\Entity\RevisionableStorageInterface $storage */
      $storage = $this->entityTypeManager->getStorage($entity_type);
      $latest_vid = $storage->getLatestRevisionId($id);
      return $storage->loadRevision($latest_vid);
    }
    catch (InvalidPluginDefinitionException | PluginNotFoundException $e) {
    }

    return NULL;
  }

  /**
   * Get the last revision log message.
   *
   * @param int $id
   *   The entity ID.
   * @param string $entity_type
   *   The entity type.
   *
   * @return string|null
   *   The revision log message, or NULL if not set.
   */
  protected function getLastRevisionLog(int $id, string $entity_type = 'node'): ?string {
    $latest_revision = $this->getLastRevision($id, $entity_type);
    if ($latest_revision instanceof RevisionLogInterface) {
      return $latest_revision->getRevisionLogMessage();
    }

    return NULL;
  }

  /**
   * Get parent entity in layout builder from context.
   *
   * @return \Drupal\Core\Entity\ContentEntityInterface|null
   *   The parent entity, or NULL if not available.
   *
   * @throws \Drupal\Component\Plugin\Exception\ContextException
   */
  protected function getMediaParentFromLayoutContext(): ?ContentEntityInterface {
    $section_storage = $this->routeMatch->getParameter('section_storage');
    // Avoid a dependency on layout_builder.
    // @phpcs:disable
    // @phpstan-ignore-next-line
    if ($section_storage instanceof \Drupal\layout_builder\Plugin\SectionStorage\OverridesSectionStorage) {
      $context = $section_storage->getContext('entity');
      if ($context) {
        try {
          $entity = $context->getContextValue();
          if ($entity instanceof ContentEntityInterface) {
            return $entity;
          }
        }
        catch (ContextException $e) {
        }
      }
    }
    return NULL;
  }

}
