<?php

namespace Drupal\revision_extras\Form;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\TypedConfigManagerInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandler;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\ConfigTarget;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Form\RedundantEditableConfigNamesTrait;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Form for configuring Revision Extras settings.
 *
 * @package Drupal\revision_extras\Form
 */
class RevisionExtrasSettingsForm extends ConfigFormBase {
  use RedundantEditableConfigNamesTrait;

  /**
   * The entityTypeManager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The module handler service.
   *
   * @var \Drupal\Core\Extension\ModuleHandler
   */
  protected ModuleHandler $moduleHandler;

  /**
   * Constructs a Revisions Extras settings form object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory service.
   * @param \Drupal\Core\Config\TypedConfigManagerInterface $typedConfigManager
   *   The typed config manager.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entityTypeManager.
   * @param \Drupal\Core\Extension\ModuleHandler $moduleHandler
   *   Module handler service.
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    TypedConfigManagerInterface $typedConfigManager,
    EntityTypeManagerInterface $entityTypeManager,
    ModuleHandler $moduleHandler,
  ) {
    parent::__construct($config_factory, $typedConfigManager);
    $this->entityTypeManager = $entityTypeManager;
    $this->moduleHandler = $moduleHandler;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory'),
      $container->get('config.typed'),
      $container->get('entity_type.manager'),
      $container->get('module_handler'),
    );
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return [
      'revision_extras.settings',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'revision_extras_settings';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $form = parent::buildForm($form, $form_state);

    $form['form_text_changes_header'] = [
      '#markup' => '<h2>' . $this->t('Revision log form appearance') . '</h2>',
    ];
    $form['form_text_changes'] = [
      '#type' => 'details',
      '#title' => $this->t('Customize form appearance'),
      '#description' => $this->t('Customizations apply in all form instances where the revision log message field appears.'),
      '#description_display' => 'before',
    ];
    $form['form_text_changes']['revision_log_message_label'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Label for the revision log message field'),
      '#config_target' => new ConfigTarget(
        'revision_extras.settings',
        'revision_log_message_label',
      ),
      '#maxlength' => 255,
      '#description' => $this->t('Leave blank to use the default label.'),
    ];
    $form['form_text_changes']['revision_log_message_description'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Description text for the revision log message field'),
      '#config_target' => new ConfigTarget(
        'revision_extras.settings',
        'revision_log_message_description',
      ),
      '#maxlength' => 255,
      '#description' => $this->t('Leave blank to use the default description.'),
    ];
    $form['form_text_changes']['append_last_log_message'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Append the last revision log message to the field description if it exists.'),
      '#config_target' => new ConfigTarget(
        'revision_extras.settings',
        'append_last_log_message',
      ),
      '#description' => $this->t('Display the last revision log message below the log message field.'),
    ];
    $form['form_text_changes']['last_log_message_label'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Label for the last revision log message in the description'),
      '#config_target' => new ConfigTarget(
        'revision_extras.settings',
        'last_log_message_label',
      ),
      '#maxlength' => 255,
      '#required' => TRUE,
      '#description' => $this->t('This only applies if appending the last revision log message to the field description.'),
      '#states' => [
        'visible' => [
          ':input[name="append_last_log_message"]' => ['checked' => TRUE],
        ],
      ],
    ];

    // Allow site builders to select which entity types require revisions and
    // log messages.
    $entity_types = [
      'node' => 'Content',
      'media' => 'Media',
      'block_content' => 'Block',
      // @todo Consider adding more revisionable types like taxonomy term.
    ];
    $node_bundle_options = [];
    $form['required_revisions'] = [
      '#type' => 'vertical_tabs',
      '#title' => '<h2>' . $this->t('Entity types with required revisions') . '</h2>',
    ];
    foreach ($entity_types as $entity_type => $entity_label) {
      // Make sure the corresponding core module is enabled before showing the
      // entity as a configuration option.
      if ($this->moduleHandler->moduleExists($entity_type)) {
        $typeStorage = $this->entityTypeManager->getStorage($entity_type . '_type');
        $entity_definition = $typeStorage->getEntityType();
        $plural_type_label = $entity_definition->getPluralLabel()->getUntranslatedString();
        $bundles = $typeStorage->loadMultiple() ?? [];

        // Create an array of bundle options for this type.
        $bundle_options = [];
        foreach ($bundles as $bundle) {
          $bundle_options[$bundle->id()] = $bundle->label();
        }
        if (!empty($bundle_options)) {
          $form['require_' . $entity_type . '_wrapper'] = [
            '#type' => 'details',
            '#title' => $this->t('@label', ['@label' => $entity_label]),
            '#description' => $this->t('@label that create a new revision on every save and require users to enter a log message.', ['@label' => ucfirst($plural_type_label)]),
            '#group' => 'required_revisions',
          ];
          $form['require_' . $entity_type . '_wrapper'][$entity_type . '_bundles'] = [
            '#type' => 'checkboxes',
            '#title' => $this->t('@label', ['@label' => ucfirst($plural_type_label)]),
            '#description' => $this->t('If none are selected, log messages will not be required on any @label.', ['@label' => $plural_type_label]),
            // Checkboxes don't accept a null default value.
            '#default_value' => $this->config('revision_extras.settings')->get($entity_type . '_bundles') ?? [],
            '#config_target' => new ConfigTarget(
              'revision_extras.settings',
              $entity_type . '_bundles',
            ),
            '#options' => $bundle_options,
          ];
          $form['require_' . $entity_type . '_wrapper']['enable_on_new_' . $entity_type] = [
            '#type' => 'checkbox',
            '#title' => $this->t('Require log message when creating new content.'),
            '#config_target' => new ConfigTarget(
              'revision_extras.settings',
              'enable_on_new_' . $entity_type,
            ),
            '#description' => $this->t('Applies to all selected types on the "add" form.'),
          ];
          if ($entity_type === 'node') {
            // Store node bundles for later use in report fields if applicable.
            $node_bundle_options = $bundle_options;
          }
          // Add option to populate default message in media library add form.
          elseif ($entity_type === 'media' && $this->moduleHandler->moduleExists('media_library')) {
            $form['require_' . $entity_type . '_wrapper']['populate_default_in_media_library'] = [
              '#type' => 'radios',
              '#title' => $this->t('Choose a method for populating a default log message when adding via the media library form. (Experimental)'),
              '#options' => [
                'none' => $this->t('Do not populate a default log message.'),
                'static' => $this->t('Static: "Added via the media library form."'),
                'parent_reference' => $this->t('Reference Parent (with fallback): "Added via the media library form in "[parent title]" (ID: [parent ID])."'),
                'parent_last_message' => $this->t('Previous revision log message of the parent entity.'),
              ],
              '#config_target' => new ConfigTarget(
                'revision_extras.settings',
                'populate_default_in_media_library',
              ),
              '#description' => $this->t('Only applies if the log message is required when adding media. Note that the parent context is not available when adding via a WYSIWYG and only exists if the parent has been saved. For media in layout builder blocks, the parent is the entity with the layout.'),
              '#states' => [
                'visible' => [
                  ':input[name="enable_on_new_media"]' => ['checked' => TRUE],
                ],
              ],
            ];
          }
        }
      }
    }

    // Configuration options for drush changed content report.
    if ($this->moduleHandler->moduleExists('node')) {
      $form['report_settings'] = [
        '#type' => 'vertical_tabs',
        '#title' => '<h2>' . $this->t('Changed content report [Experimental]') . '</h2>',
        '#description_display' => 'before',
        '#description' => $this->t('This form allows you to configure settings for a report that is triggered by a drush command. The report returns a list of public content changes for a given day and is intended for use with content moderation. To use, run the report drush command manually or schedule it via crontab. See command options using "drush help revision_extras:changed_content". Warning: supported fields may change in the future.'),
      ];
      $form['general'] = [
        '#type' => 'details',
        '#title' => $this->t('General'),
        '#group' => 'report_settings',
      ];
      $form['general']['excluded_report_bundles'] = [
        '#type' => 'checkboxes',
        '#title' => $this->t('Excluded content types'),
        '#description' => $this->t('Choose the content types to exclude from the report. If none are selected, all types will be included.'),
        // Checkboxes don't accept a null default value.
        '#default_value' => $this->config('revision_extras.settings')->get('excluded_report_bundles') ?? [],
        '#config_target' => new ConfigTarget(
          'revision_extras.settings',
          'excluded_report_bundles',
        ),
        '#options' => $node_bundle_options,
      ];
      $field_options = [
        'nid' => 'Node ID',
        'vid' => 'Version ID',
        'type' => 'Content Type',
        'title' => 'Title',
        'url_alias' => 'Url Alias',
        'publisher' => 'Publisher',
        'prev_editor' => 'Previous Editor',
        'status' => 'Status',
        'created' => 'Created',
        'modified' => 'Modified',
        'revision_link' => 'Link to Revision Diff',
        'log_message' => 'Log Message',
      ];
      // Remove the alias field if the module is not present.
      if (!$this->moduleHandler->moduleExists('path_alias')) {
        unset($field_options['url_alias']);
      }
      // Remove the diff field if the module is not present.
      if (!$this->moduleHandler->moduleExists('diff')) {
        unset($field_options['revision_link']);
      }
      $form['general']['excluded_report_fields'] = [
        '#type' => 'checkboxes',
        '#title' => $this->t('Excluded fields'),
        '#description' => $this->t('Choose the fields to exclude from the report. If none are selected, all listed fields will be included.'),
        // Checkboxes don't accept a null default value.
        '#default_value' => $this->config('revision_extras.settings')->get('excluded_report_fields') ?? [],
        '#config_target' => new ConfigTarget(
          'revision_extras.settings',
          'excluded_report_fields',
        ),
        '#options' => $field_options,
      ];
      $form['email'] = [
        '#type' => 'details',
        '#title' => $this->t('Email'),
        '#group' => 'report_settings',
      ];
      $form['email']['email_recipient'] = [
        '#type' => 'email',
        '#title' => $this->t('Primary recipient email address'),
        '#config_target' => new ConfigTarget(
          'revision_extras.settings',
          'email_recipient',
        ),
        '#description' => $this->t("The default recipient email address to be used when emailing the report. Leave empty to use the default system email address <em>(%site-email).</em>", ['%site-email' => $this->config('system.site')->get('mail')]),
        '#maxlength' => 180,
      ];
      // Group emails.
      $form['email']['email_group'] = [
        '#type' => 'textarea',
        '#rows' => 3,
        '#title' => $this->t('Additional recipient email addresses'),
        '#config_target' => new ConfigTarget(
          'revision_extras.settings',
          'email_group',
        ),
        '#description' => $this->t('Optionally send the report to these email addresses in addition to the primary recipient using the --group flag. Separate emails with commas or newlines.'),
      ];
      // Require mailsystem for mail options not supported by Drupal core.
      if ($this->moduleHandler->moduleExists('mailsystem')) {
        $form['email']['additional_options'] = [
          '#type' => 'fieldset',
          '#title' => $this->t('Options requiring additional mailer modules.'),
        ];
        $form['email']['additional_options']['email_use_html'] = [
          '#type' => 'checkbox',
          '#title' => $this->t('Send email in HTML format.'),
          '#config_target' => new ConfigTarget(
            'revision_extras.settings',
            'email_use_html',
          ),
          '#description' => $this->t('This option requires that the default configured mail handler support HTML emails. Tested with symfony_mailer_lite.'),
        ];
        // Require symfony_mailer_lite/symfony_mailer to include an attachment.
        if ($this->moduleHandler->moduleExists('symfony_mailer_lite') ||
          $this->moduleHandler->moduleExists('symfony_mailer')) {
          $form['email']['additional_options']['email_include_attachment'] = [
            '#type' => 'checkbox',
            '#title' => $this->t('Include a CSV attachment of report results on the email.'),
            '#config_target' => new ConfigTarget(
              'revision_extras.settings',
              'email_include_attachment',
            ),
            '#description' => $this->t('This option requires that the symfony_mailer_lite or symfony_mailer module be configured as the default mail handler. You may want to customize the body of the email if attaching results in a file.'),
          ];
        }
      }

      // Subject with changed content.
      $form['email']['email_report_results'] = [
        '#type' => 'fieldset',
        '#title' => $this->t('Customize email sent with changed content results.'),
      ];
      $form['email']['email_report_results']['email_report_results_subject'] = [
        '#type' => 'textfield',
        '#title' => $this->t('Subject'),
        '#config_target' => new ConfigTarget(
          'revision_extras.settings',
          'email_report_results.subject',
        ),
        '#required' => TRUE,
        '#maxlength' => 180,
        '#description' => $this->t('Accepts site-wide tokens. Use [revision_extras:report_date] for the report date.'),
      ];
      // Body with changed content.
      $form['email']['email_report_results']['email_report_results_body'] = [
        '#type' => 'textarea',
        '#title' => $this->t('Body'),
        '#config_target' => new ConfigTarget(
          'revision_extras.settings',
          'email_report_results.body',
        ),
        '#rows' => 5,
        '#description' => $this->t('Accepts site-wide tokens. Use [revision_extras:report_date] for the report date and [revision_extras:report_results] for the report results.'),
      ];
      // Subject without changed content.
      $form['email']['email_report_empty'] = [
        '#type' => 'fieldset',
        '#title' => $this->t('Customize email sent when there are NO changed content results.'),
      ];
      $form['email']['email_report_empty']['email_report_empty_subject'] = [
        '#type' => 'textfield',
        '#title' => $this->t('Subject'),
        '#config_target' => new ConfigTarget(
          'revision_extras.settings',
          'email_report_empty.subject',
        ),
        '#required' => TRUE,
        '#maxlength' => 180,
        '#description' => $this->t('Accepts site-wide tokens. Use [revision_extras:report_date] for the report date.'),
      ];
      // Body without changed content.
      $form['email']['email_report_empty']['email_report_empty_body'] = [
        '#type' => 'textarea',
        '#title' => $this->t('Body'),
        '#config_target' => new ConfigTarget(
          'revision_extras.settings',
          'email_report_empty.body',
        ),
        '#rows' => 5,
        '#description' => $this->t('Accepts site-wide tokens. Use [revision_extras:report_date] for the report date.'),
      ];
    }

    return parent::buildForm($form, $form_state);
  }

}
