<?php

namespace Drupal\revision_extras\Drush\Commands;

use Consolidation\OutputFormatters\StructuredData\RowsOfFields;
use Drupal\Component\Utility\EmailValidator;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Database\Connection;
use Drupal\Core\Datetime\DateFormatter;
use Drupal\Core\Datetime\DrupalDateTime;
use Drupal\Core\Extension\ModuleHandler;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\Language\LanguageDefault;
use Drupal\Core\Mail\MailManager;
use Drupal\Core\Render\RendererInterface;
use Drush\Attributes as CLI;
use Drush\Commands\DrushCommands;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Drush commands for revision extras module.
 */
final class RevisionExtrasCommands extends DrushCommands {

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected ConfigFactoryInterface $configFactory;

  /**
   * The email validator.
   *
   * @var \Drupal\Component\Utility\EmailValidator
   */
  protected EmailValidator $emailValidator;

  /**
   * The mail manager.
   *
   * @var \Drupal\Core\Mail\MailManager
   */
  protected MailManager $mailManager;

  /**
   * The date formatter.
   *
   * @var \Drupal\Core\Datetime\DateFormatter
   */
  protected DateFormatter $dateFormatter;

  /**
   * The file system service.
   *
   * @var \Drupal\Core\File\FileSystemInterface
   */
  protected FileSystemInterface $fileSystem;

  /**
   * Service to get default language.
   *
   * @var \Drupal\Core\Language\LanguageDefault
   */
  protected LanguageDefault $languageDefault;

  /**
   * The module handler service.
   *
   * @var \Drupal\Core\Extension\ModuleHandler
   */
  protected ModuleHandler $moduleHandler;

  /**
   * The renderer.
   *
   * @var \Drupal\Core\Render\RendererInterface
   */
  protected RendererInterface $renderer;

  /**
   * Constructs a RevisionExtrasCommands object.
   */
  public function __construct(
    private readonly Connection $connection,
    ConfigFactoryInterface $config_factory,
    EmailValidator $email_validator,
    MailManager $mail_manager,
    DateFormatter $date_formatter,
    FileSystemInterface $file_system,
    LanguageDefault $language_default,
    ModuleHandler $module_handler,
    RendererInterface $renderer,
  ) {
    parent::__construct();
    $this->configFactory = $config_factory;
    $this->emailValidator = $email_validator;
    $this->mailManager = $mail_manager;
    $this->dateFormatter = $date_formatter;
    $this->fileSystem = $file_system;
    $this->languageDefault = $language_default;
    $this->moduleHandler = $module_handler;
    $this->renderer = $renderer;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('database'),
      $container->get('config.factory'),
      $container->get('email.validator'),
      $container->get('plugin.manager.mail'),
      $container->get('date.formatter'),
      $container->get('file_system'),
      $container->get('language.default'),
      $container->get('module_handler'),
      $container->get('renderer'),
    );
  }

  /**
   * Generate a report of published/unpublished content for a given date.
   *
   * See /admin/config/user-interface/revision-extras for additional options.
   */
  #[CLI\Command(name: 'revision_extras:changed_content')]
  #[CLI\Argument(name: 'date', description: 'The report date in Y-m-d format. Defaults to today.')]
  #[CLI\Option(name: 'email', description: 'Send as email instead of displaying output.')]
  #[CLI\Option(name: 'group', description: 'Additionally send to all emails in group.')]
  #[CLI\Option(name: 'recipient', description: 'Override default recipient email.')]
  #[CLI\FieldLabels(labels: [
    'nid' => 'Node ID',
    'vid' => 'Version ID',
    'type' => 'Content Type',
    'title' => 'Title',
    'url_alias' => 'Url Alias',
    'publisher' => 'Publisher',
    'prev_editor' => 'Previous Editor',
    'status' => 'Status',
    'created' => 'Created',
    'modified' => 'Modified',
    'revision_link' => 'Link to Revision Diff',
    'log_message' => 'Log Message',
  ])]
  #[CLI\DefaultTableFields(fields: ['nid', 'title', 'status', 'modified', 'log_message'])]
  #[CLI\FilterDefaultField(field: 'title')]
  #[CLI\Usage(name: 'revision_extras:changed_content', description: 'Display a table of changed content for today.')]
  #[CLI\Usage(name: 'revision_extras:changed_content --format=json', description: 'Display json output of changed content for today.')]
  #[CLI\Usage(name: 'revision_extras:changed_content --filter="status=Unpublished"', description: 'Display a table of content that was unpublished today.')]
  #[CLI\Usage(name: 'revision_extras:changed_content 2025-09-28 --email --recipient=example@example.com --group', description: 'Instead of displaying results, send report for 2025-09-28 to example@example.com and the group.')]
  public function changedContentReport(
    string $date = '',
    array $options = [
      'format' => 'table',
      'email' => FALSE,
      'group' => FALSE,
      'recipient' => '',
    ],
  ): RowsOfFields|NULL {
    // Require the node module.
    if (!$this->moduleHandler->moduleExists('node')) {
      $this->logger()->error(dt('This command requires the node module.'));
      return NULL;
    }
    // Default to today's date using UTC.
    if (empty($date)) {
      $drupal_date = new DrupalDateTime('now', 'UTC');
      $date = $drupal_date->format('Y-m-d');
    }
    // Make sure date is in the correct format.
    elseif (!preg_match('/[0-9]{4}-[0-9]{2}-[0-9]{2}/', $date)) {
      $this->logger()->error(dt('Date is not in the correct YYYY-MM-DD format.'));
      return NULL;
    }
    // Is the report being emailed?
    $email_report = $options['email'];

    // Get module configuration.
    $report_config = $this->configFactory->get('revision_extras.settings');

    // Get attachment setting from config.
    $attachment_config = $report_config->get('email_include_attachment');
    // Requires symfony_mailer_lite or symfony_mailer module.
    $include_attachment = $attachment_config && (
      $this->moduleHandler->moduleExists('symfony_mailer_lite') ||
      $this->moduleHandler->moduleExists('symfony_mailer'));

    // Check additional options if emailing the report.
    if ($email_report) {
      // Get configured recipient with fallback to site email.
      $recipients = $report_config->get('email_recipient') ?: $this->configFactory->get('system.site')->get('mail');
      // Override default recipient email.
      if ($options['recipient']) {
        if ($this->emailValidator->isValid($options['recipient']) !== TRUE) {
          $this->logger()->error(dt('Invalid recipient email address %email', ['%email', $options['recipient']]));
          return NULL;
        }
        else {
          $recipients = $options['recipient'];
        }
      }
      // Add group to email.
      if ($options['group']) {
        // Get group list from configuration.
        $group_config = $report_config->get('email_group') ?: '';
        // Split by newline and/or comma.
        $group_emails = preg_split('/[\n,]+/', $group_config, -1, PREG_SPLIT_NO_EMPTY);
        if ($group_emails) {
          foreach ($group_emails as $key => $email) {
            $email = trim($email);
            // Remove invalid emails.
            if ($this->emailValidator->isValid($email) !== TRUE) {
              unset($group_emails[$key]);
            }
          }
          if (!empty($group_emails)) {
            $recipients .= ', ' . implode(', ', $group_emails);
          }
        }
      }
    }
    $filename = FALSE;
    $filepath = FALSE;

    // Get the site default timezone.
    $default_timezone = $this->configFactory->get('system.date')->get('timezone')['default'];

    // Convert date to a unix timestamp taking into account the site timezone.
    $unix_start = strtotime($date . ' 00:00:00 ' . $default_timezone);
    $unix_end = strtotime($date . ' 23:59:59 ' . $default_timezone);

    // Optionally exclude some content types.
    $excluded_types = $report_config->get('excluded_report_bundles') ?? [];
    $excluded_string = "'" . implode("', '", $excluded_types) . "'";

    // Only include alias if the module is enabled.
    $has_path_alias = $this->moduleHandler->moduleExists('path_alias');

    $sql = "SELECT
      cur.nid AS 'Node ID',
      cur.vid AS 'Version ID',
      prev.vid AS 'Previous ID',
      nfd.type AS 'Content Type',
      cur.title AS 'Title',
      ufd.name AS 'Publisher',
      prev_ufd.name AS 'Previous Editor',
      if(cur.status = 1, 'Published', 'Unpublished') as 'Status',
      nfd.created AS 'Created',
      cur.changed AS 'Modified',
      nr.revision_log AS 'Log Message' ";
    if ($has_path_alias) {
      $sql .= ", pa.alias AS 'URL Alias' ";
    }
    $sql .= "FROM
      {node_field_revision} cur
      INNER JOIN {node_revision} nr ON nr.vid = cur.vid
      INNER JOIN {node_field_data} nfd ON cur.nid = nfd.nid
      INNER JOIN {users_field_data} ufd ON nr.revision_uid = ufd.uid
      LEFT JOIN {node_field_revision} prev ON prev.nid = cur.nid AND prev.vid = (
        SELECT MAX(nfr2.vid) FROM node_field_revision nfr2 WHERE nfr2.nid = cur.nid AND nfr2.vid < cur.vid
      )
      LEFT JOIN {node_revision} prev_nr ON prev_nr.vid = prev.vid
      LEFT JOIN {users_field_data} prev_ufd ON prev_nr.revision_uid = prev_ufd.uid ";
    if ($has_path_alias) {
      $sql .= "LEFT JOIN {path_alias} pa ON pa.path = CONCAT('/node/', cur.nid) AND pa.id = (
          SELECT MAX(pa2.id) FROM path_alias pa2 WHERE pa2.path = CONCAT('/node/', cur.nid)
        ) ";
    }
    $sql .= "WHERE cur.changed >= $unix_start AND cur.changed <= $unix_end AND ";
    // Optionally exclude some content types as configured.
    if (!empty($excluded_types)) {
      $sql .= "nfd.type NOT IN ($excluded_string) AND ";
    }
    $sql .= "( cur.status = 1 OR (prev.status = 1 AND cur.status = 0 AND nfd.status = 0))
      ORDER BY cur.nid, cur.vid;";

    // Get all nodes that have a public change for provided date.
    $query = $this->connection->prepareStatement($sql, [], TRUE);
    $query->execute();

    // This should match the command's "FieldLabels".
    $available_fields = [
      'nid' => 'Node ID',
      'vid' => 'Version ID',
      'type' => 'Content Type',
      'title' => 'Title',
      'url_alias' => 'Url Alias',
      'publisher' => 'Publisher',
      'prev_editor' => 'Previous Editor',
      'status' => 'Status',
      'created' => 'Created',
      'modified' => 'Modified',
      'revision_link' => 'Link to Revision Diff',
      'log_message' => 'Log Message',
    ];
    // Remove the alias column if the module is not present.
    if (!$this->moduleHandler->moduleExists('path_alias')) {
      unset($available_fields['url_alias']);
    }
    // Remove the diff column if the module is not present.
    if (!$this->moduleHandler->moduleExists('diff')) {
      unset($available_fields['revision_link']);
    }
    // Limit fields in report output.
    $excluded_fields = $report_config->get('excluded_report_fields') ?? [];
    // Remove excluded fields.
    if (!empty($excluded_fields)) {
      foreach ($available_fields as $key => $label) {
        if (in_array($key, $excluded_fields)) {
          unset($available_fields[$key]);
        }
      }
    }
    // Header for results.
    $result_header = array_values($available_fields);

    // Array for result rows.
    $table_rows = [];

    // If there are results, return them.
    if ($query->rowCount() > 0) {
      // Create a CSV file if emailing the report with an attachment.
      if ($email_report && $include_attachment) {
        // Write the file to the temporary file directory.
        $filename = "revision-extras-changed-content-$date.csv";
        $schema = 'temporary://';
        $filepath = $this->fileSystem->realpath($schema . $filename);
        $fp = fopen($filepath, 'w');
        fputcsv($fp, $result_header, escape: '\\');
      }

      // Add rows to results table and optionally CSV file.
      while ($row = $query->fetchAssoc()) {
        // @todo For embedded types, get parent node info from entity_usage if
        // available.
        if ($row['Previous ID'] && $this->moduleHandler->moduleExists('diff')) {
          // Create a link comparing each revision with its previous revision.
          $latest_revision_link = sprintf('/node/%s/revisions/view/%s/%s/visual_inline',
            $row['Node ID'],
            $row['Previous ID'],
            $row['Version ID'],
          );
        }

        // Format created and modified dates using site timezone.
        $formatted_created = $this->dateFormatter->format($row['Created'], 'custom', 'Y-m-d h:i A', $default_timezone);
        $formatted_modified = $this->dateFormatter->format($row['Modified'], 'custom', 'Y-m-d h:i A', $default_timezone);

        // Add a row to the results.
        $table_row = [
          'nid' => $row['Node ID'],
          'vid' => $row['Version ID'],
          'type' => $row['Content Type'],
          'title' => $row['Title'],
          'url_alias' => $row['URL Alias'] ?? '',
          'publisher' => $row['Publisher'],
          'prev_editor' => $row['Previous Editor'] ?? '',
          'status' => $row['Status'],
          'created' => $formatted_created,
          'modified' => $formatted_modified,
          'revision_link' => $latest_revision_link ?? '',
          'log_message' => trim($row['Log Message']),
        ];
        // Remove the alias column if the module is not present.
        if (!$this->moduleHandler->moduleExists('path_alias')) {
          unset($table_row['url_alias']);
        }
        // Remove the diff column if the module is not present.
        if (!$this->moduleHandler->moduleExists('diff')) {
          unset($table_row['revision_link']);
        }
        // Remove excluded fields.
        if (!empty($excluded_fields)) {
          foreach ($table_row as $key => $label) {
            if (in_array($key, $excluded_fields)) {
              unset($table_row[$key]);
            }
          }
        }
        $table_rows[] = $table_row;

        // Add a row to the file if emailing the report with an attachment.
        if ($email_report && $include_attachment && isset($fp)) {
          fputcsv($fp, array_values($table_row), escape: '\\');
        }
      }

      // Close the file if emailing an attachment.
      if ($email_report && $include_attachment && isset($fp)) {
        fclose($fp);
      }
    }

    if ($email_report && isset($recipients)) {
      $params['context']['is_html'] = FALSE;
      $params['context']['report_date'] = $date;

      // Create simple message if there were no content changes.
      if (empty($table_rows)) {
        $params['context']['report_results'] = '';
        // Use config with tokens for subject and message.
        $params['context']['subject'] = $report_config->get('email_report_empty.subject') ?: '';
        $params['context']['message'] = $report_config->get('email_report_empty.body') ?: '';
      }
      // Optionally attach file with results and/or include directly in body.
      else {
        // Populate custom report_data token with results.
        $table_row_values = [];
        foreach ($table_rows as $row) {
          $table_row_values[] = array_values($row);
        }
        // Get configuration setting.
        $html_config = $report_config->get('email_use_html');
        // Requires mailsystem module.
        $html_format = $html_config && $this->moduleHandler->moduleExists('mailsystem');

        if ($html_format) {
          // For HTML email, render a table.
          $render_array = [
            '#type' => 'table',
            '#header' => $result_header,
            '#rows' => $table_row_values,
            '#attributes' => [
              'border' => '1',
              'cellspacing' => '0',
              'cellpadding' => '0',
            ],
          ];
          $params['context']['report_results'] = $this->renderer->renderInIsolation($render_array);
          $params['context']['is_html'] = TRUE;
        }
        else {
          // Convert the results into a plain text "table".
          $params['context']['report_results'] = $this->arrayToText($table_row_values, $result_header);
        }

        // Use config with tokens for subject and message.
        $params['context']['subject'] = $report_config->get('email_report_results.subject') ?: '';
        $params['context']['message'] = $report_config->get('email_report_results.body') ?: '';
        if ($include_attachment) {
          $params['attachments'][] = [
            'filename' => $filename,
            'filepath' => $filepath,
            'filemime' => 'text/csv',
          ];
        }
      }

      try {
        // Use the site default language.
        $langcode = $this->languageDefault->get()->getId();
        $emailResult = $this->mailManager->mail('revision_extras', 'changed_content', $recipients, $langcode, $params);
        if ($emailResult['result'] !== TRUE) {
          $this->logger->error(dt('Result from server: !error', ['!error' => $emailResult]));
        }
        else {
          if (!empty($table_rows)) {
            $this->output()->writeln(dt('Sent content changes for %date to %email.', [
              '%date' => $date,
              '%email' => $recipients,
            ]));
          }
          else {
            $this->output()->writeln(dt('No content changes for %date.', ['%date' => $date]));
          }
          $this->logger()->success(dt('Content report email for %date sent to %email successfully!', [
            '%date' => $date,
            '%email' => $recipients,
          ]));
        }
      }
      catch (\Exception $e) {
        $this->logger->error(dt('Failed to send content report email to %email. Exception: !error', [
          '%email' => $recipients,
          '!error' => $e->getMessage(),
        ]));
      }

      // Delete the CSV file to clean up.
      if ($include_attachment && file_exists($filepath)) {
        unlink($filepath);
      }
      return NULL;
    }

    // Display report output.
    if (!empty($table_rows)) {
      return new RowsOfFields($table_rows);
    }
    else {
      $this->output()->writeln(dt('No content changes for %date.', ['%date' => $date]));
      return NULL;
    }
  }

  /**
   * Convert an array into a text-based table.
   *
   * @param array $data
   *   Array with data.
   * @param array $header
   *   Header row values.
   * @param string $separator
   *   Column separator.
   *
   * @return string
   *   Array converted to a text-based table string.
   */
  private function arrayToText(array $data, array $header = [], string $separator = ' | '): string {
    $output = [];
    $column_widths = [];

    // Calculate maximum column widths.
    foreach ($data as $row) {
      foreach ($row as $col_index => $value) {
        $column_widths[$col_index] = max($column_widths[$col_index] ?? 0, mb_strlen((string) $value));
      }
    }

    // Include header widths in calculation if provided.
    if (!empty($header)) {
      foreach ($header as $col_index => $value) {
        $column_widths[$col_index] = max($column_widths[$col_index] ?? 0, mb_strlen((string) $value));
      }
    }

    // Format header row.
    if (!empty($header)) {
      $formatted_header_cols = [];
      foreach ($header as $col_index => $value) {
        $formatted_header_cols[] = str_pad((string) $value, $column_widths[$col_index]);
      }
      $output[] = implode($separator, $formatted_header_cols);
      // Add a separator line below the header.
      $output[] = str_repeat('-', array_sum($column_widths) + (count($column_widths) - 1) * mb_strlen($separator));
    }

    // Format data rows with padding to create "columns".
    foreach ($data as $row) {
      $formatted_row_cols = [];
      foreach ($row as $col_index => $value) {
        $formatted_row_cols[] = str_pad((string) $value, $column_widths[$col_index]);
      }
      $output[] = implode($separator, $formatted_row_cols);
    }

    // Separate rows with newlines.
    return implode("\n", $output);
  }

}
