<?php

declare(strict_types=1);

namespace Drupal\Tests\revision_extras\Functional;

use Drupal\Tests\BrowserTestBase;

/**
 * Test customized revision log message form field.
 *
 * @group revision_extras
 */
class CustomLogMessageFieldTest extends BrowserTestBase {

  /**
   * User with admin privileges.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $adminUser;

  /**
   * The installation profile.
   *
   * @var string
   */
  protected $profile = 'standard';

  /**
   * The default theme.
   *
   * @var string
   */
  protected $defaultTheme = 'stark';

  /**
   * The modules to load to run the test.
   *
   * @var array
   */
  protected static $modules = ['user', 'block_content', 'media', 'node', 'revision_extras'];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Create admin user.
    $this->adminUser = $this->drupalCreateUser([
      'administer site configuration',
      'create page content',
      'edit any page content',
      'create media',
      'create basic block content',
      'access block library',
    ]);

    // Login.
    $this->drupalLogin($this->adminUser);
  }

  /**
   * Test with default field label and description.
   */
  public function testWithDefaultFieldText(): void {
    // Get page node add form.
    $this->drupalGet('node/add/page');

    // Check that revision log message exists with default label.
    $this->assertSession()->fieldExists('Revision log message');

    // Check that revision log message description matches the default.
    $this->assertSession()->pageTextContains('Briefly describe the changes you have made.');

    $title = 'Page with default log message text';
    $initial_message = 'Created initial page node unique revision message';
    $this->submitForm([
      'title[0][value]' => $title,
      'revision_log[0][value]' => $initial_message,
    ], 'Save');

    $node = $this->drupalGetNodeByTitle($title);
    $this->drupalGet('node/' . $node->id() . '/edit');

    // Make sure the last revision log message does not appear.
    $this->assertSession()->pageTextNotContains($initial_message);
  }

  /**
   * Test configuring revision extra options.
   */
  protected function testSettingsForm(): void {
    // Configure settings form.
    $this->drupalGet('/admin/config/user-interface/revision-extras');

    // Assure we loaded the settings form.
    $this->assertSession()->statusCodeEquals(200);

    // Customize the label and description.
    $this->submitForm([
      'revision_log_message_label' => 'Unique custom label',
      'revision_log_message_description' => 'Unique custom description',
      'append_last_log_message' => TRUE,
      'last_log_message_label' => 'Last custom label',
    ], 'Save configuration');

    $this->assertSession()->pageTextContains('The configuration options have been saved.');
  }

  /**
   * Test with customized field label and description.
   */
  public function testWithCustomizedFieldText(): void {
    $this->testSettingsForm();
    $label = 'Unique custom label';
    $description = 'Unique custom description';

    // Get page node add form.
    $this->drupalGet('node/add/page');

    // Check that revision log message label is customized.
    $this->assertSession()->fieldExists($label);

    // Check that revision log message description is customized.
    $this->assertSession()->pageTextContains($description);

    $title = 'Page with customized log message text';
    $initial_message = 'Created initial page node unique revision message';
    $this->submitForm([
      'title[0][value]' => $title,
      'revision_log[0][value]' => $initial_message,
    ], 'Save');

    $node = $this->drupalGetNodeByTitle($title);
    $this->drupalGet('node/' . $node->id() . '/edit');

    // Make sure the last revision log message appears.
    $this->assertSession()->pageTextContains($initial_message);

    // Get image media add form.
    $this->drupalGet('media/add/image');

    // Check that revision log message label is customized.
    $this->assertSession()->fieldExists($label);

    // Check that revision log message description is customized.
    $this->assertSession()->pageTextContains($description);

    // Get basic block add form.
    $this->drupalGet('block/add/basic');

    // Check that revision log message label is customized.
    $this->assertSession()->fieldExists($label);

    // Check that revision log message description is customized.
    $this->assertSession()->pageTextContains($description);
  }

}
