<?php

declare(strict_types=1);

namespace Drupal\Tests\revision_extras\Functional;

use Drupal\Tests\BrowserTestBase;

/**
 * Test requiring revisions/revision log message.
 *
 * @group revision_extras
 */
class RequireLogMessageFieldTest extends BrowserTestBase {

  /**
   * User with admin privileges.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $adminUser;

  /**
   * The installation profile.
   *
   * @var string
   */
  protected $profile = 'standard';

  /**
   * The default theme.
   *
   * @var string
   */
  protected $defaultTheme = 'stark';

  /**
   * The modules to load to run the test.
   *
   * @var array
   */
  protected static $modules = ['user', 'block_content', 'node', 'revision_extras'];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Create admin user.
    $this->adminUser = $this->drupalCreateUser([
      'administer site configuration',
      'create page content',
      'edit any page content',
      'view all revisions',
      'create basic block content',
      'edit any basic block content',
      'view any basic block content history',
      'access block library',
    ]);

    // Login.
    $this->drupalLogin($this->adminUser);
  }

  /**
   * Test node without required revisions.
   */
  public function testNodeNotRequired(): void {
    // Get page node add form.
    $this->drupalGet('node/add/page');

    // Check that revision log message field exists with default label.
    $this->assertSession()->fieldExists('Revision log message');

    // Confirm we can add a page without a revision log message.
    $title = 'Page with no log message text';
    $this->submitForm(['title[0][value]' => $title], 'Save');

    $node = $this->drupalGetNodeByTitle($title);

    // Edit the page creating a revision without a revision log message.
    $this->drupalGet('node/' . $node->id() . '/edit');
    $updated_title = 'Updated page with no log message text';
    $this->submitForm([
      'title[0][value]' => $updated_title,
      'revision' => TRUE,
    ], 'Save');

    // Check that there are 2 revisions.
    $this->drupalGet('node/' . $node->id() . '/revisions');
    $rows = $this->xpath('//table/tbody/tr');
    $this->assertCount(2, $rows, 'The table contains two revisions.');

    // Edit the page again without making another revision.
    $this->drupalGet('node/' . $node->id() . '/edit');
    $this->submitForm([
      'title[0][value]' => 'One more time',
      'revision' => FALSE,
    ], 'Save');

    // Check that there are still only 2 revisions.
    $this->drupalGet('node/' . $node->id() . '/revisions');
    $rows = $this->xpath('//table/tbody/tr');
    $this->assertCount(2, $rows, 'The table still contains two revisions.');
  }

  /**
   * Test block without required revisions.
   */
  public function testBlockNotRequired(): void {
    // Get basic block add form.
    $this->drupalGet('block/add/basic');

    // Check that revision log message field exists with default label.
    $this->assertSession()->fieldExists('Revision log message');

    // Confirm we can add a block without a revision log message.
    $label = 'Block without required revisions';
    $this->submitForm(['info[0][value]' => $label], 'Save');

    $blocks = \Drupal::entityTypeManager()
      ->getStorage('block_content')
      ->loadByProperties(['info' => $label]);
    $block = reset($blocks);

    // Edit the block again.
    $this->drupalGet('admin/content/block/' . $block->id());
    $updated_label = 'Updated block without required revisions';
    $this->submitForm(['info[0][value]' => $updated_label], 'Save');

    $this->assertSession()->pageTextContains('Basic block ' . $updated_label . ' has been updated.');

    // Check that there is only one revision.
    $this->drupalGet('admin/content/block/' . $block->id() . '/revisions');
    $rows = $this->xpath('//table/tbody/tr');
    $this->assertCount(1, $rows, 'The table only contains one revision.');
  }

  /**
   * Test configuring revision extra options.
   */
  protected function testSettingsForm(array $values): void {
    // Get the settings form.
    $this->drupalGet('/admin/config/user-interface/revision-extras');

    // Assure we loaded the settings form.
    $this->assertSession()->statusCodeEquals(200);

    // Update the settings.
    $this->submitForm($values, 'Save configuration');

    $this->assertSession()->pageTextContains('The configuration options have been saved.');
  }

  /**
   * Test node form with required revisions.
   */
  public function testNodeWithRequired(): void {
    // Require revisions on page, image, and basic block.
    $this->testSettingsForm(['node_bundles[page]' => TRUE]);

    // Get page node add form.
    $this->drupalGet('node/add/page');

    $title = 'Initial page - no required message on new';
    $this->submitForm([
      'title[0][value]' => $title,
    ], 'Save');

    $node = $this->drupalGetNodeByTitle($title);

    // Update the page and try to save without a revision.
    $this->drupalGet('node/' . $node->id() . '/edit');

    // Check that the revision checkbox is not displayed.
    $this->assertSession()->fieldNotExists('Create new revision');
    // Check that the revision log message is displayed.
    $this->assertSession()->fieldExists('Revision log message');

    $updated_title = 'Updated page - no required message on new';
    $this->submitForm([
      'title[0][value]' => $updated_title,
    ], 'Save');

    // Returns an error due to empty revision log message.
    $this->assertSession()->pageTextNotContains('Basic page ' . $updated_title . ' has been updated.');

    // Add a log message and resubmit.
    $this->submitForm([
      'title[0][value]' => $updated_title,
      'revision_log[0][value]' => 'Edited . ' . $updated_title,
    ], 'Save');

    $this->assertSession()->pageTextContains('Basic page ' . $updated_title . ' has been updated.');

    // Update settings so that log message is required for new pages.
    $this->testSettingsForm(['enable_on_new_node' => TRUE]);

    // Get page node add form again.
    $this->drupalGet('node/add/page');

    $title = 'Second page - with required message on new';
    $this->submitForm([
      'title[0][value]' => $title,
    ], 'Save');

    // Returns an error due to empty revision log message.
    $this->assertSession()->pageTextNotContains('Basic page ' . $title . ' has been created.');
  }

  /**
   * Test block form with required revisions.
   */
  public function testBlockWithRequired(): void {
    // Require revisions on page, image, and basic block.
    $this->testSettingsForm(['block_content_bundles[basic]' => TRUE]);

    // Get basic block add form.
    $this->drupalGet('block/add/basic');

    // Confirm we can add a block without a revision log message.
    $label = 'Block - no revisions on new';
    $this->submitForm(['info[0][value]' => $label], 'Save');

    $blocks = \Drupal::entityTypeManager()
      ->getStorage('block_content')
      ->loadByProperties(['info' => $label]);
    $block = reset($blocks);

    // Edit the block again.
    $this->drupalGet('admin/content/block/' . $block->id());
    $updated_label = 'Updated block - no revisions on new';
    $this->submitForm(['info[0][value]' => $updated_label], 'Save');

    // Returns an error because log message is required.
    $this->assertSession()->pageTextNotContains('Basic block ' . $updated_label . ' has been updated.');

    // Add a log message and resubmit.
    $this->submitForm([
      'info[0][value]' => $updated_label,
      'revision_log[0][value]' => 'Edited . ' . $updated_label,
    ], 'Save');

    // Check that there are two revisions.
    $this->drupalGet('admin/content/block/' . $block->id() . '/revisions');
    $rows = $this->xpath('//table/tbody/tr');
    $this->assertCount(2, $rows, 'The table contains two revisions.');

    // Update settings so that log message is required for new blocks.
    $this->testSettingsForm(['enable_on_new_block_content' => TRUE]);

    // Get basic block add form again.
    $this->drupalGet('block/add/basic');

    $label = 'Second block - with required message on new';
    $this->submitForm(['info[0][value]' => $label], 'Save');

    // Returns an error due to empty revision log message.
    $this->assertSession()->pageTextNotContains('Basic page ' . $label . ' has been created.');
  }

}
