<?php

namespace Drupal\revision_manager\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\revision_manager\RevisionManagerBatchInterface;
use Drupal\Core\Config\TypedConfigManagerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Form\SubformState;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\revision_manager\Plugin\RevisionManagerPluginManager;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Plugin\PluginFormInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\revision_manager\EntityHelper;
use Drupal\revision_manager\Manager\EntityManager;
use Drupal\Core\Config\Entity\ConfigEntityInterface;

/**
 * Provides a configuration form for Revision Manager settings.
 */
final class AdminSettingsForm extends ConfigFormBase {

  public function __construct(
    ConfigFactoryInterface $configFactory,
    TypedConfigManagerInterface $typedConfigManager,
    protected RevisionManagerPluginManager $pluginManager,
    protected EntityTypeManagerInterface $entityTypeManager,
    protected EntityHelper $entityHelper,
    protected EntityManager $entityManager,
    protected FormHelper $formHelper,
    protected RevisionManagerBatchInterface $batch,
  ) {
    parent::__construct($configFactory, $typedConfigManager);
  }

  /**
   * Creates an instance of the AdminSettingsForm.
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('config.factory'),
      $container->get('config.typed'),
      $container->get('plugin.manager.revision_manager'),
      $container->get('entity_type.manager'),
      $container->get('revision_manager.entity_helper'),
      $container->get('revision_manager.entity_manager'),
      $container->get('revision_manager.form_helper'),
      $container->get('revision_manager.batch'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'revision_manager_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['revision_manager.settings'];
  }

  /**
   * Provides a summary of the settings for a given entity type and bundle.
   *
   * @param string $entity_type_id
   *   The entity type ID.
   * @param string $bundle_id
   *   The bundle ID.
   *
   * @return string
   *   A string indicating whether the settings are overridden or default.
   */
  private function getSettingsSummary(string $entity_type_id, string $bundle_id): string {
    return $this->isOverridden($entity_type_id, $bundle_id)
      ? $this->t('Overridden')
      : $this->t('Default');
  }

  /**
   * Returns TRUE if this entity‑bundle has overridden settings.
   */
  private function isOverridden(string $entity_type_id, string $bundle_id): bool {
    $settings = $this->entityManager->getSettingsEntityTypeBundle($entity_type_id, $bundle_id);
    return isset($settings['status']) && $settings['status'] === 'overridden';
  }

  /**
   * Builds a drop button link to the edit form, if available.
   *
   * @param \Drupal\Core\Config\Entity\ConfigEntityInterface $bundle
   *   The bundle entity.
   *
   * @return array
   *   A render array for the dropbutton or an empty array if not applicable.
   */
  private function getDropButton(ConfigEntityInterface $bundle): array {
    // Check if there is an edit form for the bundle entity.
    if (!$bundle->hasLinkTemplate('edit-form')) {
      return [];
    }

    // Build and check the URL from the edit-form link template.
    $url = $bundle->toUrl('edit-form');
    if (!$url->isRouted()) {
      return [];
    }

    return [
      '#links' => [
        'edit' => [
          'title' => $this->t('Configure'),
          'url' => $url,
        ],
      ],
      '#type' => 'dropbutton',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('revision_manager.settings');
    $plugin_definitions = $this->pluginManager->getDefinitions();

    // Generate the form for each entity type.
    foreach ($this->entityHelper->getSupportedEntityTypes() as $entity_type_id) {
      $definition = $this->entityTypeManager->getDefinition($entity_type_id);
      if ($bundle_type = $definition->getBundleEntityType()) {

        $form[$entity_type_id] = [
          '#open' => FALSE,
          '#title' => $definition->getLabel() ?: $entity_type_id,
          '#tree' => TRUE,
          '#type' => 'details',
        ];

        $entity_type_enabled = $this->entityManager->isEntityTypeEnabled($entity_type_id);
        $form[$entity_type_id]['enable_entity_type'] = [
          '#type' => 'checkbox',
          '#title' => $this->t('Enabled'),
          '#description' => $this->t('Enable Revision Manager for this entity type.'),
          '#default_value' => $entity_type_enabled,
        ];

        $form[$entity_type_id]['description'] = [
          '#tag' => 'p',
          '#type' => 'html_tag',
          '#value' => $this->t('Below are the defaults for each plugin, which will be applied to all bundles for this entity type which do not have overridden settings.'),
        ];

        $form[$entity_type_id]['settings'] = [
          '#type' => 'container',
          '#states' => [
            'visible' => [
              ':input[name="' . $entity_type_id . '[enable_entity_type]"]' => ['checked' => TRUE],
            ],
          ],
        ];

        $bundles = $this->entityTypeManager->getStorage($bundle_type)->loadMultiple();
        $rows = [];
        foreach ($bundles as $bundle) {
          if (!$bundle instanceof ConfigEntityInterface) {
            continue;
          }

          $bundle_id = $bundle->id();
          $settings_summary = $this->getSettingsSummary($entity_type_id, $bundle_id);
          $drop_button = $this->getDropButton($bundle);
          $overridden = $this->isOverridden($entity_type_id, $bundle_id);

          $rows[] = [
            'data' => [
              $bundle->label(),
              $bundle_id,
              $settings_summary,
              ['data' => $drop_button],
            ],
            'class' => [
              $overridden ? 'color-warning' : '',
            ],
          ];
        }

        // Add each plugin for the entity type.
        foreach ($plugin_definitions as $plugin_id => $definition) {
          $settings = $this->pluginManager->getDefaultPluginSettings($plugin_id, $entity_type_id);
          $form[$entity_type_id]['settings'][$plugin_id] = $this->formHelper->buildPluginSubform(
            $entity_type_id,
            $plugin_id,
            $definition,
            $settings,
            $form,
            $form_state,
          );
        }

        $header = [
          ['data' => $this->t('Bundle')],
          ['data' => $this->t('Machine name'), 'class' => [RESPONSIVE_PRIORITY_MEDIUM]],
          ['data' => $this->t('Settings'), 'class' => [RESPONSIVE_PRIORITY_MEDIUM]],
          ['data' => $this->t('Operations')],
        ];

        $form[$entity_type_id]['settings']['table'] = [
          '#header' => $header,
          '#rows' => $rows,
          '#type' => 'table',
          '#states' => [
            'visible' => [
              ':input[name="' . $entity_type_id . '[enable_entity_type]"]' => ['checked' => TRUE],
            ],
          ],
          '#empty' => $this->t('No supported entity types available.'),
        ];
      }

    }

    $form['verbose_log'] = [
      '#default_value' => $config->get('verbose_log') ?? FALSE,
      '#description' => $this->t('Record revision deletion activities in watchdog.'),
      '#title' => $this->t('Log activities to watchdog.'),
      '#type' => 'checkbox',
    ];

    $form['disable_automatic_queueing'] = [
      '#default_value' => $config->get('disable_automatic_queueing') ?? FALSE,
      '#description' => $this->t('If checked, entities will not be automatically added to the queue for processing when a new revision is added.'),
      '#title' => $this->t('Disable automatic queueing'),
      '#type' => 'checkbox',
    ];

    $form['enqueue_entities_now'] = $this->formHelper->enqueueNowForm();

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $config = $this->config('revision_manager.settings');
    $defaults = [];

    $plugin_definitions = $this->pluginManager->getDefinitions();

    foreach ($this->entityHelper->getSupportedEntityTypes() as $entity_type_id) {
      $enable_entity_type = (bool) $form_state->getValue([$entity_type_id, 'enable_entity_type']);
      $enabled_entities = $config->get('enabled_entities') ?? [];
      $enabled_entities[$entity_type_id] = $enable_entity_type;
      $config->set('enabled_entities', $enabled_entities);

      foreach ($plugin_definitions as $plugin_id => $plugin_definition) {
        $plugin = $this->pluginManager->getPlugin($plugin_id, $entity_type_id);
        if ($plugin instanceof PluginFormInterface) {
          if (isset($form[$entity_type_id]['settings'][$plugin_id])) {
            $sub_form_state = SubformState::createForSubform($form[$entity_type_id]['settings'][$plugin_id], $form, $form_state);
            $defaults[$entity_type_id][$plugin_id] = $sub_form_state->getValues();
          }
          else {
            $defaults[$entity_type_id][$plugin_id] = $this->pluginManager->getDefaultPluginSettings($plugin_id, $entity_type_id);
          }
        }

      }
    }

    $config->set('defaults', $defaults);
    $config->set('disable_automatic_queueing', $form_state->getValue('disable_automatic_queueing'));
    $config->set('verbose_log', $form_state->getValue('verbose_log'));
    $config->save();

    if ($form_state->getValue('enqueue_entities_now')) {
      $this->batch->run();
    }

    parent::submitForm($form, $form_state);
  }

}
