<?php

namespace Drupal\revision_manager\Plugin;

use Drupal\Component\Plugin\PluginBase;
use Drupal\Core\DependencyInjection\DependencySerializationTrait;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Entity\TranslatableInterface;
use Drupal\Core\Entity\Query\QueryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Language\LanguageInterface;
use Drupal\Core\Entity\RevisionableInterface;
use Drupal\Core\Entity\FieldableEntityInterface;

/**
 * Base class for Revision Manager plugins.
 */
abstract class RevisionManagerBase extends PluginBase implements RevisionManagerInterface {

  use StringTranslationTrait;
  use DependencySerializationTrait;

  /**
   * Default chunk size for revision ids.
   */
  private const REVISION_CHUNK_SIZE = 500;

  /**
   * {@inheritdoc}
   */
  public function __construct(
    array $configuration,
    string $plugin_id,
    mixed $plugin_definition,
    private readonly EntityTypeManagerInterface $entityTypeManager,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): self {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('entity_type.manager'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getConfiguration(): array {
    return $this->configuration;
  }

  /**
   * {@inheritdoc}
   */
  public function setConfiguration(array $configuration): void {
    $this->configuration = $configuration + $this->defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [];
  }

  /**
   * {@inheritdoc}
   */
  public function validateConfigurationForm(array &$form, FormStateInterface $form_state): void {
    // Default implementation does no validation.
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state): void {
    $this->configuration[$this->getPluginId()] = $form_state->getValues();
  }

  /**
   * Delete revisions for a given entity.
   */
  abstract public function deleteRevisions(RevisionableInterface $entity): array;

  /**
   * Build a query to retrieve all revisions for a given entity.
   *
   * @param \Drupal\Core\Entity\RevisionableInterface $entity
   *   The entity for which deletable revision IDs are being retrieved.
   *
   * @return \Drupal\Core\Entity\Query\QueryInterface
   *   The built query object.
   */
  protected function buildRevisionQuery(RevisionableInterface $entity): QueryInterface {
    $storage = $this->entityTypeManager->getStorage($entity->getEntityTypeId());
    $revisionKey = $entity->getEntityType()->getKey('revision');

    // The entity doesn’t store multiple revisions (e.g., config entity).
    // Skip sorting or bail out early.
    if (!$revisionKey) {
      return $storage->getQuery()->accessCheck(FALSE);
    }

    return $storage->getQuery()
      ->allRevisions()
      ->accessCheck(FALSE)
      ->condition($entity->getEntityType()->getKey('id'), $entity->id())
      ->sort($revisionKey, 'DESC');
  }

  /**
   * Retrieve revision IDs in chunks from a base query.
   *
   * This chunking prevents memory issues when dealing with large datasets when
   * trying to load allRevisions() in one go.
   *
   * @param \Drupal\Core\Entity\Query\QueryInterface $query
   *   The base query to execute.
   * @param int $chunk_size
   *   Size of chunks to process.
   *
   * @return int[]
   *   Array of revision IDs.
   */
  protected function collectRevisionIds(QueryInterface $query, int $chunk_size = self::REVISION_CHUNK_SIZE): array {
    $revision_ids = [];
    $offset = 0;

    do {
      $chunk = (clone $query)
        ->range($offset, $chunk_size)
        ->execute();

      $revision_ids = array_merge($revision_ids, array_keys($chunk));
      $offset += $chunk_size;
    } while ($chunk !== []);

    return $revision_ids;
  }

  /**
   * Return revision IDs per language, newest-first, that match SQL conditions.
   *
   * @param \Drupal\Core\Entity\RevisionableInterface $entity
   *   The entity for which to return revision IDs.
   * @param array<int,array{field:string,value:mixed,operator?:string}> $conditions
   *   Extra query conditions.
   *
   * @return array<string,int[]>
   *   Revision IDs, keyed by langcode.
   */
  protected function getRevisionIds(RevisionableInterface $entity, array $conditions = []): array {
    $revision_ids = [];

    $langcodes = $entity instanceof TranslatableInterface
      ? array_keys($entity->getTranslationLanguages())
      : [LanguageInterface::LANGCODE_NOT_SPECIFIED];

    foreach ($langcodes as $langcode) {
      $query = $this->buildRevisionQuery($entity);

      if ($langcode !== LanguageInterface::LANGCODE_NOT_SPECIFIED) {
        $query->condition('langcode', $langcode);
      }
      foreach ($conditions as $c) {
        $query->condition($c['field'], $c['value'], $c['operator'] ?? '=');
      }

      $revision_ids[$langcode] = $this->collectRevisionIds($query);
    }

    return $revision_ids;
  }

  /**
   * Calculates the revision IDs that can be safely deleted.
   *
   * Always protects the entity’s current revision.
   *
   * @param \Drupal\Core\Entity\RevisionableInterface $entity
   *   The parent entity (provides the current revision ID).
   * @param array<string,int[]> $buckets
   *   Revision-ID arrays keyed by langcode, newest-first.
   * @param int $keep
   *   Additional revisions to keep. Always protects the current revision.
   *
   * @return int[]
   *   Flat list of revision IDs safe to delete.
   */
  protected function getRemovableRevisionIds(RevisionableInterface $entity, array $buckets, int $keep = 0): array {
    $current = (int) $entity->getRevisionId();
    $offset = $keep > 0 ? $keep - 1 : 0;
    $to_delete = [];

    foreach ($buckets as $ids) {
      $without_current = array_filter($ids, static fn (int $id): bool => $id !== $current);
      $to_prune = array_slice($without_current, $offset);
      $to_delete = array_merge($to_delete, $to_prune);
    }

    // Protect against default revision IDs.
    $to_delete = array_filter($to_delete, static fn (int $id): bool => $id !== $current);
    return array_values(array_unique($to_delete));
  }

  /**
   * Determine the field name that indicates the publication state of an entity.
   *
   * This method provides a consistent way to identify whether an entity is
   * "published" by accommodating variations of status fields across different
   * entity types.
   *
   * @param \Drupal\Core\Entity\RevisionableInterface $entity
   *   The entity to check.
   *
   * @return string
   *   The machine name of the field that controls publication status.
   */
  protected function getStatusField(RevisionableInterface $entity): string {
    $key = $entity->getEntityType()->getKey('published') ?: $entity->getEntityType()->getKey('status');
    return $key ?: ($entity instanceof FieldableEntityInterface && $entity->hasField('enabled') ? 'enabled' : NULL);
  }

}
