<?php

namespace Drupal\Tests\revision_manager\Kernel;

use Drupal\Core\Entity\RevisionableInterface;

/**
 * Tests combined Age + Amount plugins of Revision Manager.
 *
 * @group revision_manager
 */
class IntegrationPluginTest extends RevisionManagerTestBase {

  /**
   * Verify that both Age and Amount plugins work together on nodes.
   */
  public function testAgeAndAmountPluginsNode(): void {
    $this->configureRevisionManager('node', 'age', TRUE, ['age' => 6]);
    $this->configureRevisionManager('node', 'amount', TRUE, ['amount' => 3]);

    // Create the base node.
    $node = $this->createEntity('node', ['type' => 'page', 'title' => 'Age / Amount Integration Test', 'status' => 1]);
    $this->assertInstanceOf(RevisionableInterface::class, $node);

    $cutoff = (new \DateTimeImmutable())->modify('-6 months')->getTimestamp();

    // Add 5 revisions older than the cutoff and 6 newer ones.
    $this->createRevisionsAroundCutoff($node, 5, 6, $cutoff, self::ONE_DAY * 10, self::ONE_DAY);
    $this->processRevisionQueue(self::QUEUE_ID);

    // Verify exactly 3 revisions (current + 2 previous) to remain.
    $this->assertRevisionCount($node, 3);
  }

  /**
   * Verify drafts remain when Age and Amount plugins run together.
   */
  public function testAgeAndAmountSkipUnpublished(): void {
    $this->configureRevisionManager('node', 'age', TRUE, ['age' => 6]);
    $this->configureRevisionManager('node', 'amount', TRUE, ['amount' => 2]);

    $node = $this->createEntity('node', [
      'type' => 'page',
      'title' => 'Age / Amount Unpublished Integration Test',
      'status' => 1,
    ]);
    $this->assertInstanceOf(RevisionableInterface::class, $node);

    $this->createRevisions($node, 3);
    $this->createRevisions($node, 2, ['status' => 0]);

    $this->processRevisionQueue(self::QUEUE_ID);

    // Verify that 2 published + 2 drafts = 4 revisions left.
    $this->assertRevisionCount($node, 4);
  }

}
