<?php

namespace Drupal\Tests\revision_manager\Kernel;

/**
 * Tests queue generation logic for Revision Manager.
 *
 * @group revision_manager
 */
class QueueGeneratorTest extends RevisionManagerTestBase {

  /**
   * The queue generator service.
   *
   * @var \Drupal\revision_manager\Manager\QueueGenerator
   */
  protected $generator;

  /**
   * The queue for remove revisions.
   *
   * @var \Drupal\Core\Queue\QueueInterface
   */
  protected $queue;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Enable the node entity type for queue tests.
    $this->configureRevisionManager('node', 'amount', TRUE);

    $this->generator = $this->container->get('revision_manager.queue_generator');
    $this->queue = $this->container->get('queue')->get(self::QUEUE_ID);

    // Clear any existing queue items and state.
    $this->generator->clearQueue();
  }

  /**
   * Test enqueueEntity adds an item only once per entity.
   */
  public function testEnqueueEntityIsUnique(): void {
    $entity = $this->createEntity('node', ['type' => 'page', 'title' => 'Unique', 'status' => 1]);
    $id = (string) $entity->id();

    // First enqueue should succeed.
    $this->assertTrue($this->generator->enqueueEntity('node', $id));
    $this->assertEquals(1, $this->queue->numberOfItems());

    // Second enqueue should be ignored.
    $this->assertFalse($this->generator->enqueueEntity('node', $id));
    $this->assertEquals(1, $this->queue->numberOfItems());
  }

  /**
   * Test clearQueue empties the queue and resets the state.
   */
  public function testClearQueue(): void {
    $entity = $this->createEntity('node', ['type' => 'page', 'title' => 'Clear', 'status' => 1]);
    $id = (string) $entity->id();
    $this->generator->enqueueEntity('node', $id);

    $this->generator->clearQueue();
    $this->assertEquals(0, $this->queue->numberOfItems());

    // After clearing, enqueue should succeed again.
    $this->assertTrue($this->generator->enqueueEntity('node', $id));
  }

  /**
   * Test enqueueRevisionsByBundle enqueues all entities in a bundle.
   */
  public function testEnqueueByBundle(): void {
    // Create multiple nodes in the 'page' bundle.
    $this->createEntity('node', ['type' => 'page', 'title' => 'One', 'status' => 1]);
    $this->createEntity('node', ['type' => 'page', 'title' => 'Two', 'status' => 1]);

    $this->generator->enqueueRevisionsByBundle('node', 'page');
    $this->assertEquals(2, $this->queue->numberOfItems());
  }

  /**
   * Test enqueueRevisionsByType enqueues all entities across bundles.
   */
  public function testEnqueueByType(): void {
    // Create nodes in the 'page' and 'article' bundle.
    $this->createEntity('node', ['type' => 'page', 'title' => 'Page Bundle', 'status' => 1]);
    $this->createEntity('node', ['type' => 'article', 'title' => 'Article', 'status' => 1]);

    $this->generator->enqueueRevisionsByType('node');
    $this->assertEquals(2, $this->queue->numberOfItems());
  }

  /**
   * Test disabled entity types are not enqueued.
   */
  public function testDisabledEntityTypeDoesNotEnqueue(): void {
    // Disable the node entity type in config.
    $this->configureRevisionManager('node', 'amount', FALSE);

    $entity = $this->createEntity('node', ['type' => 'page', 'title' => 'Not Queued', 'status' => 1]);
    $this->assertFalse($this->generator->enqueueEntity('node', (string) $entity->id()));
    $this->assertEquals(0, $this->queue->numberOfItems());
  }

}
