<?php

namespace Drupal\revision_manager\Plugin\RevisionManager;

use Drupal\Core\Entity\RevisionableInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\revision_manager\Plugin\RevisionManagerBase;

/**
 * Deletes older revisions beyond the specified age.
 *
 * @RevisionManager(
 *   id = "age",
 *   label = @Translation("Delete revisions older than a specified age"),
 *   description = @Translation(
 *     "Deletes older revisions beyond the specified age while always
 *      preserving active and unpublished revisions."
 *   )
 * )
 */
final class Age extends RevisionManagerBase {
  /**
   * Month buckets for the “age” dropdown.
   */
  private const AGE_OPTIONS = [1, 3, 6, 12, 24, 36, 48];

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $options = [];
    foreach (self::AGE_OPTIONS as $months) {
      $options[$months] = $this->formatPlural($months, '@count month', '@count months');
    }

    $form['age'] = [
      '#type' => 'select',
      '#title' => $this->t('Minimum age of revisions to keep'),
      '#description' => $this->t('Only revisions older than the selected age will be removed. Active and unpublished revisions are always preserved.'),
      '#options' => $options,
      '#default_value' => $this->configuration['age'] ?? 6,
      '#required' => TRUE,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function deleteRevisions(RevisionableInterface $entity): array {
    $cutoff = (new \DateTimeImmutable())
      ->modify(sprintf('-%d months', (int) ($this->configuration['age'] ?? 6)))
      ->getTimestamp();

    $buckets = $this->getRevisionIds($entity, [
      ['field' => $this->getStatusField($entity), 'value' => 1],
      ['field' => 'changed', 'value' => $cutoff, 'operator' => '<'],
    ]);

    // Get eligible revisions for deletion.
    return $this->getRemovableRevisionIds($entity, $buckets, 0);
  }

}
