<?php

namespace Drupal\revision_manager;

use Drupal\Core\Batch\BatchBuilder;
use Drupal\Core\DependencyInjection\DependencySerializationTrait;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\revision_manager\Manager\EntityManager;
use Drupal\revision_manager\Manager\QueueGenerator;

/**
 * Batch controller to enqueue entities for revision cleanup.
 */
final class RevisionManagerBatch implements RevisionManagerBatchInterface {
  use StringTranslationTrait;
  use DependencySerializationTrait;

  public function __construct(
    private readonly QueueGenerator $queueGenerator,
    private readonly LoggerChannelFactoryInterface $logger,
    private readonly MessengerInterface $messenger,
    private readonly EntityHelper $entityHelper,
    private readonly EntityManager $entityManager,
  ) {}

  /**
   * {@inheritdoc}
   */
  public function run(string $from = RevisionManagerBatchInterface::ENQUEUE_TYPE_FORM): bool {
    $this->queueGenerator->clearQueue();

    [$operations, $total_items] = $this->buildOperations();

    // If no entities to process.
    if ($operations === []) {
      return FALSE;
    }

    // Setup and run the batch.
    $batch = $this->setupBatch($operations, $total_items);

    if ($from === RevisionManagerBatchInterface::ENQUEUE_TYPE_DRUSH) {
      $batch =& batch_get();
      $batch['progressive'] = FALSE;
      drush_backend_batch_process();
    }

    return TRUE;
  }

  /**
   * Set up the batch with operations.
   *
   * @param array<array{0: callable, 1: array<mixed>}> $operations
   *   Operations to be processed.
   * @param int $total_items
   *   Total number of items to process.
   *
   * @return array
   *   The batch array.
   */
  private function setupBatch(array $operations, int $total_items): array {
    array_unshift($operations, [
      [$this, 'initBatch'],
      [$total_items],
    ]);

    // Build the batch.
    $builder = (new BatchBuilder())
      ->setTitle($this->t('Enqueuing entities for revision cleanup'))
      ->setInitMessage($this->t('Initializing @count entities', ['@count' => $total_items]))
      ->setProgressMessage($this->t('Processed @current of @total chunks'))
      ->setErrorMessage($this->t('An error occurred while enqueueing entities'))
      ->setFinishCallback([$this, 'finishBatch']);

    // Add each chunk operation.
    foreach ($operations as [$callback, $args]) {
      $builder->addOperation($callback, $args);
    }

    // Dispatch the batch.
    $batch = $builder->toArray();
    batch_set($batch);

    return $batch;
  }

  /**
   * {@inheritdoc}
   */
  private function buildOperations(): array {
    $operations = [];
    $total_items = 0;

    foreach ($this->entityHelper->getSupportedEntityTypes() as $type) {
      if (!$this->entityManager->isEntityTypeEnabled($type)) {
        continue;
      }

      $ids = $this->entityHelper->getEntityIds($type);
      $total_items += count($ids);

      foreach (array_chunk($ids, self::BATCH_SIZE) as $chunk) {
        $operations[] = [
          [$this, 'processEntityChunk'],
          [$type, $chunk],
        ];
      }
    }

    return [$operations, $total_items];
  }

  /**
   * Seeds the batch context with our total-item count.
   *
   * @param int $total_items
   *   The total number of items to be processed in the batch.
   * @param array<string,mixed> $context
   *   Batch context array, which stores information about the batch process.
   */
  public function initBatch(int $total_items, array &$context): void {
    $context['results']['total'] = 0;
    $context['results']['max'] = $total_items;
  }

  /**
   * {@inheritdoc}
   */
  public function processEntityChunk(string $entity_type_id, array $entity_ids, array &$context): void {
    foreach ($entity_ids as $id) {
      if ($this->queueGenerator->enqueueEntity($entity_type_id, $id)) {
        $context['results']['total']++;
        $context['finished'] = $context['results']['total'] / $context['results']['max'];
      }
    }

    $context['message'] = $this->formatProgressMessage(
      $context['results']['total'],
      $context['results']['max'],
      $entity_type_id,
    );
  }

  /**
   * Formats the progress message based on environment.
   *
   * @param int $current
   *   Current number of processed items.
   * @param int $total
   *   Total number of items to process.
   * @param string $type
   *   The entity type being processed.
   *
   * @return string
   *   The formatted message.
   */
  private function formatProgressMessage(int $current, int $total, string $type): string {
    $args = [
      '@current' => $current,
      '@total' => $total,
      '@type' => $type,
    ];

    if (PHP_SAPI === 'cli') {
      return $this->t('Queued @current of @total entities | type: @type', $args);
    }

    return $this->t('Queued @current of @total entities<br>Current entity type: <em>@type</em>', $args);
  }

  /**
   * {@inheritdoc}
   */
  public function finishBatch(bool $success, array $results, array $operations): bool {
    $logger = $this->logger->get('revision_manager');
    $total = $results['total'] ?? 0;

    if ($success && $total > 0) {
      $message = $this->formatPlural(
        $total,
        'One entity queued for cleanup.',
        '@count entities queued for cleanup.',
        ['@count' => $total],
      );
      $logger->info($message);
      $this->messenger->addMessage($message);
    }
    elseif (!$success) {
      $msg = $this->t('Revision queueing did not complete.');
      $logger->error($msg);
      $this->messenger->addError($msg);
    }

    return $success;
  }

}
